<?php

declare(strict_types=1);

namespace Insight\Component\Hydrator;

use Insight\Component\Hydrator\Exception\HydratorConfigurationException;
use Insight\Component\Hydrator\Type\ArrayType;
use Insight\Component\Hydrator\Type\BooleanType;
use Insight\Component\Hydrator\Type\FloatType;
use Insight\Component\Hydrator\Type\IntegerType;
use Insight\Component\Hydrator\Type\JsonType;
use Insight\Component\Hydrator\Type\Mongo\BinaryType;
use Insight\Component\Hydrator\Type\Mongo\DateTimeImmutableType;
use Insight\Component\Hydrator\Type\Mongo\DateTimeType;
use Insight\Component\Hydrator\Type\Mongo\IdType;
use Insight\Component\Hydrator\Type\NumberType;
use Insight\Component\Hydrator\Type\StringType;

final class TypeCollection
{
    /**
     * @var array<string, Type>
     */
    private array $storage = [];

    public static function basic(): self
    {
        $instance = new self()
            ->add(new ArrayType())
            ->add(new BooleanType())
            ->add(new FloatType())
            ->add(new IntegerType())
            ->add(new JsonType())
            ->add(new NumberType())
            ->add(new StringType());

        if (extension_loaded('mongodb')) {
            $instance
                ->add(new BinaryType())
                ->add(new DateTimeImmutableType())
                ->add(new DateTimeType())
                ->add(new IdType());
        }

        return $instance;
    }

    public function add(Type $type): self
    {
        $this->storage[$type::class] = $type;

        foreach ($type->names() as $typeName) {
            $this->storage[$typeName] = $type;
        }

        return $this;
    }

    public function get(string $typeName): Type
    {
        return $this->storage[$typeName]
            ?? throw new HydratorConfigurationException("Unsupported type: $typeName");
    }

    public function has(string $typeName): bool
    {
        return array_key_exists($typeName, $this->storage);
    }
}
