<?php

declare(strict_types=1);

namespace Insight\Component\Hydrator;

use Insight\Component\Hydrator\Exception\HydratorConfigurationException;
use Insight\Component\Hydrator\Exception\HydratorException;
use Insight\Component\Hydrator\Exception\HydratorExceptionType;
use LogicException;
use ReflectionClass;
use ReflectionException;
use TypeError;

final readonly class Hydrator
{
    public function __construct(
        private SchemaCollection $schemaCollection,
        private TypeCollection $typeCollection,
    ) {
    }

    /**
     * @template T
     *
     * @param class-string<T> $type
     *
     * @return T
     *
     * @throws HydratorException
     */
    public function hydrate(string $type, array $data)
    {
        $schema = $this->schemaCollection->get($type);

        if (!class_exists($type)) {
            throw new LogicException();
        }

        $reflection = new ReflectionClass($type);
        $properties = [];
        $errors = [];

        foreach ($schema->fields as $field) {
            $value = $data[$field->name] ?? null;

            if (null === $value) {
                if (!$field->isNullable) {
                    $errors[] = HydratorExceptionType::NOT_NULLABLE->raise($field->name);
                    continue;
                }
            }

            if (null !== $value) {
                try {
                    $value = $this->hydrateProperty($field, $value);
                } catch (HydratorException $exception) {
                    $errors[] = $exception->changeField($field->name);
                    continue;
                }
            }

            $properties[$field->propertyName] = $value;
        }

        if (count($errors) !== 0) {
            throw HydratorExceptionType::ERROR->raise(errors: $errors);
        }

        try {
            $instance = $reflection->newInstanceWithoutConstructor();
        } catch (ReflectionException) {
            throw new HydratorConfigurationException("Could not instantiate the class: $type");
        }

        foreach ($properties as $name => $value) {
            try {
                $property = $reflection->getProperty($name);
            } catch (ReflectionException) {
                throw new HydratorConfigurationException("Property does not exist: $type::$$name");
            }

            try {
                $property->setValue($instance, $value);
            } catch (TypeError) {
                throw new HydratorConfigurationException("Property type is inconsistent with the hydrator type: $type::$$name");
            }
        }

        return $instance;
    }

    private function hydrateProperty(Field $field, mixed $value): mixed
    {
        if (!$field->isArray) {
            return $this->hydrateValue($field, $value);
        }

        if (!is_array($value)) {
            throw HydratorExceptionType::EXPECTED_ARRAY->raise();
        }

        $errors = [];
        $values = [];

        foreach ($value as $index => $element) {
            try {
                $values[] = $this->hydrateValue($field, $element);
            } catch (HydratorException $exception) {
                $errors[] = $exception->changeField((string) $index);
            }
        }

        if (count($errors) !== 0) {
            throw HydratorExceptionType::ERROR->raise(errors: $errors);
        }

        return $values;
    }

    private function hydrateValue(Field $field, mixed $value): mixed
    {
        if ($field->isEnum) {
            $value = call_user_func([$field->type, 'tryFrom'], $value);

            return null !== $value
                ? $value
                : throw HydratorExceptionType::EXPECTED_ENUM->raise();
        }

        if ($field->isDocument && !$this->typeCollection->has($field->type)) {
            if (!is_array($value)) {
                throw HydratorExceptionType::EXPECTED_OBJECT->raise();
            }
            return $this->hydrate($field->type, $value);
        }

        return $this->typeCollection->get($field->type)->parse($value, $this->typeCollection);
    }
}
