<?php

declare(strict_types=1);

namespace Insight\Component\Hydrator;

use DirectoryIterator;
use Insight\Component\Hydrator\Exception\HydratorConfigurationException;
use Insight\Component\Hydrator\Loader\FileLoader;
use SplFileInfo;

final readonly class SchemaCollectionLoader
{
    /**
     * @param array<string, FileLoader> $loaders
     */
    private function __construct(private array $loaders)
    {
    }

    /**
     * @param iterable<FileLoader> $loaders
     */
    public static function create(iterable $loaders): self
    {
        $map = [];

        foreach ($loaders as $loader) {
            if (!$loader instanceof FileLoader) {
                throw new HydratorConfigurationException('Loader must implement: ' . FileLoader::class);
            }
            foreach ($loader->extensions() as $extension) {
                $map[$extension] = $loader;
            }
        }

        return new self($map);
    }

    public function load(SchemaCollection $collection, string $path): void
    {
        switch (true) {
            case is_dir($path):
                $this->directory($collection, new DirectoryIterator($path));
                break;

            case is_file($path):
                $this->file($collection, new SplFileInfo($path));
                break;

            default:
                throw new HydratorConfigurationException("Invalid path: $path");
        }
    }

    private function directory(SchemaCollection $collection, DirectoryIterator $directory): void
    {
        if (!$directory->isReadable()) {
            throw new HydratorConfigurationException("Path is not readable: {$directory->getRealPath()}");
        }

        foreach ($directory as $directoryItem) {
            if ($directoryItem->isDot()) {
                continue;
            }

            if ($directoryItem->isDir()) {
                $this->directory($collection, new DirectoryIterator($directoryItem->getRealPath()));
                continue;
            }

            $this->file($collection, $directoryItem);
        }
    }

    private function file(SchemaCollection $collection, SplFileInfo $file): void
    {
        if (!$file->isReadable()) {
            throw new HydratorConfigurationException("Path is not readable: {$file->getRealPath()}");
        }

        $loader = $this->loaders[$file->getExtension()] ?? null;

        if (null === $loader) {
            return;
        }

        $loader->load($file, $collection);
    }
}
