<?php

declare(strict_types=1);

namespace Insight\Component\EventSourcing;

use Insight\Component\EventSourcing\Exception\UniqueConstraintException;

/**
 * @template TEvent of Event
 */
interface EventStore extends Listenable
{
    /**
     * Adds another index that is going to react on every appended event.
     *
     * Event store MUST ensure that indexes are updated within THE SAME transaction as events are appended.
     */
    public function addIndex(Index $index): void;

    /**
     * Appends set of records to the stream as new events.
     *
     * This method SHOULD defer the actual operation, and the caller has to call "commit()" to store the events.
     *
     * @param Record[] $records
     */
    public function append(string $streamId, int $streamRevision, array $records): void;

    /**
     * Commits current transaction by appending all scheduled events to the store.
     *
     * @throws UniqueConstraintException When attempted to insert a unique index that already exists.
     */
    public function commit(): void;

    /**
     * Performs any operation necessary for the implementation to work flawlessly.
     *
     * This method and it's usage is strictly implementation-specific, therefore it SHOULD be properly documented.
     */
    public function initialize(): void;

    /**
     * Returns an event stream by its unique identifier.
     *
     * @return EventStream<TEvent>
     */
    public function stream(string $streamId): EventStream;

    /**
     * Returns an event stream with all the events in the event log.
     *
     * @return EventStream<TEvent>
     */
    public function streamAll(): EventStream;
}
