<?php

declare(strict_types=1);

namespace Insight\Component\EventSourcing\Mongo;

use DateTimeImmutable;
use Doctrine\ODM\MongoDB\DocumentManager;
use Doctrine\ODM\MongoDB\MongoDBException;
use Generator;
use Insight\Component\EventSourcing\EventStore;
use Insight\Component\EventSourcing\Exception\UnexpectedRevisionException;
use Insight\Component\EventSourcing\Mongo\Doctrine\CustomDriver;
use Insight\Component\EventSourcing\Mongo\Document\Event;
use Insight\Component\EventSourcing\Mongo\Document\Position;
use Insight\Component\EventSourcing\Mongo\Document\System;
use Insight\Component\EventSourcing\Mongo\Value\Guid;
use Insight\Component\EventSourcing\Mongo\Value\Time;
use Insight\Component\EventSourcing\Record;
use InvalidArgumentException;
use LogicException;
use Throwable;

/**
 * @template-implements EventStore<Event>
 */
final readonly class MongoEventStore implements EventStore
{
    use CustomDriver;

    public function __construct(private DocumentManager $documentManager)
    {
        $this->loadMetadataDriver($this->documentManager);
    }

    /**
     * @throws MongoDBException
     */
    public function append(string $streamId, int $streamRevision, array $records): void
    {
        $currentRevision = $this->stream($streamId)->currentRevision();

        if ($streamRevision !== $currentRevision) {
            throw new UnexpectedRevisionException($streamId, $streamRevision, $currentRevision);
        }

        foreach ($records as $record) {
            $record instanceof Record || throw new InvalidArgumentException();

            $this->documentManager->persist(
                new Event(
                    id: Guid::generate()->toString(),
                    time: Time::now()->milliseconds(),
                    type: $record->type,
                    version: $record->version,
                    streamId: $streamId,
                    streamRevision: ++$streamRevision,
                    systemRevision: $this->reserveSystemVersion(),
                    meta: $record->meta,
                    data: $record->data,
                ),
            );
        }
    }

    /**
     * @throws Throwable
     */
    public function commit(): void
    {
        $this->documentManager->flush();
    }

    public function initialize(): void
    {
        $schemaManager = $this->documentManager->getSchemaManager();

        $schemaManager->ensureDocumentIndexes(Event::class);
        $schemaManager->ensureDocumentIndexes(Position::class);
        $schemaManager->ensureDocumentIndexes(System::class);
    }

    public function listen(?int $fromRevision = null): Generator
    {
        return $this->streamAll()->listen($fromRevision);
    }

    public function stream(string $streamId): MongoEventStream
    {
        return new MongoEventStream($this->documentManager, $streamId);
    }

    public function streamAll(): MongoEventStream
    {
        return new MongoEventStream($this->documentManager);
    }

    /**
     * @throws MongoDBException
     */
    private function reserveSystemVersion(): int
    {
        $systemInfo = $this->documentManager
            ->getRepository(System::class)
            ->createQueryBuilder()
                ->findAndUpdate()
                ->returnNew()
                ->upsert()
                ->field('_id')->equals('System')
                ->field('revision')->inc(1)
                ->field('createdAt')->setOnInsert(new DateTimeImmutable())
                ->field('updatedAt')->set(new DateTimeImmutable())
            ->getQuery()
            ->execute();

        if (!$systemInfo instanceof System) {
            throw new LogicException();
        }

        return $systemInfo->revision;
    }
}
