<?php

declare(strict_types=1);

namespace Insight\Component\EventSourcing\Metadata\Info;

use Insight\Component\EventSourcing\Attribute\ProjectionEventHandler;
use Insight\Component\EventSourcing\Event;
use Insight\Component\EventSourcing\Exception\DefinitionException;
use Insight\Component\EventSourcing\Metadata;
use Insight\Component\EventSourcing\Projection;
use InvalidArgumentException;
use ReflectionClass;
use ReflectionNamedType;

/**
 * @internal
 *
 * @phpstan-type HandlerList array<class-string, non-empty-list<non-empty-string>>
 */
final class ProjectionInfo
{
    /**
     *
     * @param HandlerList $handlers
     */
    private function __construct(public array $handlers)
    {
    }

    public static function parse(string $type): self
    {
        if (!class_exists($type) || !is_a($type, Projection::class, true)) {
            throw new InvalidArgumentException();
        }

        $reflection = new ReflectionClass($type);

        return new self(
            self::resolveHandlers($reflection),
        );
    }

    /**
     * @return list<non-empty-string>
     */
    public function handlersByType(string $type): array
    {
        return $this->handlers[$type] ?? [];
    }

    /**
     * @param ReflectionClass<Projection> $reflection
     *
     * @return HandlerList
     */
    private static function resolveHandlers(ReflectionClass $reflection): array
    {
        $handlers = [];

        foreach ($reflection->getMethods() as $method) {
            if (count($method->getAttributes(ProjectionEventHandler::class)) === 0) {
                continue;
            }

            if (!$method->isProtected()) {
                throw new DefinitionException('Projection event handler must be a protected method.');
            }

            $parameters = $method->getParameters();

            if (count($parameters) !== 2) {
                throw new DefinitionException('Projection event handler must accept exactly two arguments.');
            }

            $type1 = $parameters[0]->getType();
            $type2 = $parameters[1]->getType();

            if (!$type1 instanceof ReflectionNamedType) {
                throw new DefinitionException('Projection event handler must take a valid aggregate event as a first parameter');
            }

            try {
                $eventInfo = Metadata::get()->aggregateEvent($type1->getName());
            } catch (InvalidArgumentException) {
                throw new DefinitionException('Projection event handler must take a valid aggregate event as a first parameter');
            }

            if (!$type2 instanceof ReflectionNamedType || $type2->getName() !== Event::class) {
                throw new DefinitionException('Projection event handler must take an Event as a second argument.');
            }

            $handlers[$eventInfo->type][] = $method->getName();
        }

        return $handlers;
    }
}
