<?php

declare(strict_types=1);

namespace Insight\Component\EventSourcing;

use InvalidArgumentException;
use LogicException;
use ReflectionClass;
use ReflectionException;
use RuntimeException;

final class EventSourceAggregateRepository
{
    /**
     * @var array<string, EventSourceAggregate>
     */
    private array $aggregatesUnitOfWork = [];

    public function __construct(private readonly EventStore $store)
    {
    }

    public function exists(string $aggregateId): bool
    {
        return !$this->store->stream($aggregateId)->isEmpty();
    }

    /**
     * @template TAggregate of EventSourceAggregate
     *
     * @param class-string<TAggregate> $aggregateType
     *
     * @return ?EventSourceAggregate
     * @return ?TAggregate
     *
     * @throws ReflectionException
     */
    public function load(string $aggregateId, string $aggregateType): ?EventSourceAggregate
    {
        $baseReflection = new ReflectionClass(EventSourceAggregate::class);
        $applyFn = $baseReflection->getMethod('apply');

        $aggregate = $this->aggregatesUnitOfWork[$aggregateId] ?? null;

        if (null !== $aggregate) {
            if (!is_a($aggregate, $aggregateType, true)) {
                throw new InvalidArgumentException();
            }
            return $aggregate;
        }

        $stream = $this->store->stream($aggregateId);

        if ($stream->isEmpty()) {
            return null;
        }

        $metadata = \Insight\Component\EventSourcing\Metadata::get();
        $metadata->aggregate($aggregateType);

        $aggregate = new ReflectionClass($aggregateType)->newInstanceWithoutConstructor();

        $aggregateVersion = 0;

        foreach ($stream->all() as $event) {
            $eventInfo = $metadata->aggregateEventByName($event->type(), $event->version());

            if (null === $eventInfo) {
                throw new RuntimeException("Event not registered -> {name: {$event->type()}, version: {$event->version()}}");
            }

            $aggregateEvent = new ($eventInfo->type)(...$event->data());

            $applyFn->invoke($aggregate, $aggregateEvent);

            $v = $event->streamRevision();

            if ($v !== $aggregateVersion + 1) {
                throw new InvalidArgumentException('SOMETHING WENT TERRIBLY WRONG!');
            }

            $aggregateVersion = $v;
        }

        $baseReflection->getProperty('aggregateId')->setValue($aggregate, $aggregateId);
        $baseReflection->getProperty('aggregateVersion')->setValue($aggregate, $aggregateVersion);

        return $this->aggregatesUnitOfWork[$aggregate->aggregateId] = $aggregate;
    }

    public function reset(): void
    {
        $this->aggregatesUnitOfWork = [];
    }

    /**
     * @param list<EventSourceAggregate> $untracked
     */
    public function save(array $untracked = []): void
    {
        $this->track($untracked);

        $reflection = new ReflectionClass(EventSourceAggregate::class);

        $updates = [];

        foreach ($this->aggregatesUnitOfWork as $aggregate) {
            if (count($aggregate->aggregateChanges) === 0) {
                continue;
            }

            $this->store->append(
                $aggregate->aggregateId,
                $aggregate->aggregateVersion,
                array_map(
                    $this->createRecordFromAggregateChange(...),
                    $aggregate->aggregateChanges,
                ),
            );

            $updates[] = $aggregate;
        }

        if (empty($updates)) {
            return;
        }

        $this->store->commit();

        foreach ($updates as $aggregate) {
            $stream = $this->store->stream($aggregate->aggregateId);

            $reflection->getProperty('aggregateChanges')->setValue($aggregate, []);
            $reflection->getProperty('aggregateVersion')->setValue($aggregate, $stream->currentRevision());
        }
    }

    /**
     * @param mixed[] $aggregates
     */
    public function track(array $aggregates): void
    {
        foreach ($aggregates as $aggregate) {
            if (!$aggregate instanceof EventSourceAggregate) {
                return;
            }

            $persisted = $this->aggregatesUnitOfWork[$aggregate->aggregateId] ?? null;

            if (null !== $persisted) {
                if ($persisted !== $aggregate) {
                    throw new LogicException('Different instance of the same aggregate is already being tracked.');
                }
                return;
            }

            if ($this->exists($aggregate->aggregateId)) {
                throw new LogicException('An aggregate with the same ID already exists.');
            }

            $id = $aggregate->aggregateId;

            $this->aggregatesUnitOfWork[$id] = $aggregate;
        }
    }

    /**
     * @param array{object, mixed[]} $change
     */
    private function createRecordFromAggregateChange(array $change): Record
    {
        [$object, $meta] = $change;

        $type = $object::class;
        $info = Metadata::get()->aggregateEvent($type);

        $data = (array) $object;

        return new Record(
            $info->name,
            $info->version,
            $meta,
            $data,
        );
    }
}
