<?php

declare(strict_types=1);

namespace Insight\Component\EventSourcing;

use Insight\Component\EventSourcing\Metadata\Info\AggregateEventInfo;
use Insight\Component\EventSourcing\Metadata\Info\AggregateInfo;
use Insight\Component\EventSourcing\Metadata\Info\ProjectionInfo;
use InvalidArgumentException;

/**
 * @internal
 */
final class Metadata
{
    /**
     * @var self
     */
    private static Metadata $instance;

    /**
     * @var array<string, AggregateInfo>
     */
    private array $aggregateCache = [];

    /**
     * @var array<string, AggregateEventInfo>
     */
    private array $aggregateEventCache = [];

    /**
     * @var array<string, ProjectionInfo>
     */
    private array $projectionCache = [];

    private function __construct()
    {
    }

    public static function get(): self
    {
        return self::$instance ??= new self();
    }

    public function aggregate(string $type): AggregateInfo
    {
        return $this->aggregateCache[$type] ??= $this->parseAggregateInfo($type);
    }

    public function aggregateEvent(string $type): AggregateEventInfo
    {
        return $this->aggregateEventCache[$type] ??= $this->parseAggregateEventInfo($type);
    }

    public function aggregateEventByName(string $name, int $version): ?AggregateEventInfo
    {
        return $this->aggregateEventCache["$name@$version"] ?? null;
    }

    public function projection(string $type): ProjectionInfo
    {
        return $this->projectionCache[$type] ??= $this->parseProjectionInfo($type);
    }

    private function parseAggregateEventInfo(string $type): AggregateEventInfo
    {
        $info = AggregateEventInfo::parse($type);

        $nameKey = "$info->name@$info->version";

        if (array_key_exists($nameKey, $this->aggregateEventCache)) {
            throw new InvalidArgumentException("Multiple aggregate event declarations for: $nameKey");
        }

        $this->aggregateEventCache[$nameKey] = $info;

        return $info;
    }

    private function parseAggregateInfo(string $type): AggregateInfo
    {
        $info = AggregateInfo::parse($type);

        foreach ($info->handlers as $event => $_) {
            $this->aggregateEvent($event); // preload aggregate events
        }

        return $info;
    }

    private function parseProjectionInfo(string $type): ProjectionInfo
    {
        return ProjectionInfo::parse($type);
    }
}
