<?php

declare(strict_types=1);

namespace Insight\Component\EventSourcing\Mongo;

use Doctrine\ODM\MongoDB\DocumentManager;
use Doctrine\ODM\MongoDB\Iterator\Iterator;
use Doctrine\ODM\MongoDB\MongoDBException;
use Doctrine\ODM\MongoDB\Query\Builder;
use Generator;
use Insight\Component\EventSourcing\EventStream;
use Insight\Component\EventSourcing\Mongo\Document\Event;
use Insight\Component\EventSourcing\Mongo\Value\Time;
use MongoDB\BSON\Timestamp;

/**
 * @template-implements EventStream<Event>
 * @internal
 */
final readonly class MongoEventStream implements EventStream
{
    public function __construct(private DocumentManager $documentManager, private ?string $streamId = null)
    {
    }

    /**
     * @throws MongoDBException
     */
    public function above(int $revision): Iterator
    {
        return $this
            ->createQuery()
                ->field($this->revisionField())->gt($revision)
            ->getQuery()
            ->getIterator();
    }

    /**
     * @throws MongoDBException
     */
    public function all(): Iterator
    {
        return $this
            ->createQuery()
            ->getQuery()
            ->getIterator();
    }

    /**
     * @throws MongoDBException
     */
    public function count(): int
    {
        return $this
            ->createQuery()
            ->count()
            ->getQuery()
            ->execute();
    }

    public function currentRevision(): int
    {
        $event = $this
            ->createQuery()
            ->sort('SystemRevision', 'DESC')
            ->getQuery()
            ->getSingleResult();

        if (!$event instanceof Event) {
            return 0;
        }

        return null === $this->streamId
             ? $event->systemPosition()
             : $event->streamPosition();
    }

    /**
     * @throws MongoDBException
     */
    public function isEmpty(): bool
    {
        return $this->count() === 0;
    }

    public function listen(?int $fromRevision = null): Generator
    {
        $timeFrom = Time::now()->milliseconds();

        if (null !== $fromRevision) {
            yield from $this->above($fromRevision);
        }

        $pipeline = [];

        if (null !== $this->streamId) {
            $pipeline[] = [
                '$match' => [
                    'fullDocument.StreamId' => $this->streamId,
                    'operationType' => 'insert',
                ],
            ];
        }

        $pipeline[] = [
            '$project' => [
                'documentKey' => 1,
            ],
        ];

        $stream = $this
            ->documentManager
            ->getDocumentCollection(Event::class)
            ->watch($pipeline, [
                'startAtOperationTime' => new Timestamp(0, Time::fromMilliseconds($timeFrom)->timestamp()),
            ]);

        $repository = $this->documentManager->getRepository(Event::class);

        for ($stream->rewind(); true; $stream->next()) {
            if (!$stream->valid()) {
                continue;
            }

            $event = $stream->current();

            if (!is_array($event)) {
                continue;
            }

            yield $repository->find($event['documentKey']['_id']);
        }
    }

    /**
     * @throws MongoDBException
     */
    public function until(int $revision): Iterator
    {
        return $this
            ->createQuery()
                ->field($this->revisionField())->lte($revision)
            ->getQuery()
            ->getIterator();
    }

    private function createQuery(): Builder
    {
        $query = $this->documentManager
            ->getRepository(Event::class)
            ->createQueryBuilder()
            ->sort('SystemRevision', 'ASC');

        if (null !== $this->streamId) {
            $query->field('StreamId')->equals($this->streamId);
        }

        return $query;
    }

    private function revisionField(): string
    {
        return null === $this->streamId ? 'SystemRevision' : 'StreamRevision';
    }
}
