<?php

declare(strict_types=1);

namespace Insight\Component\EventSourcing\Metadata\Info;

use Insight\Component\EventSourcing\Attribute\AggregateEmit;
use Insight\Component\EventSourcing\Attribute\AggregateEventHandler;
use Insight\Component\EventSourcing\Attribute\AggregateName;
use Insight\Component\EventSourcing\EventSourceAggregate;
use InvalidArgumentException;
use LogicException;
use ReflectionClass;
use ReflectionNamedType;

final readonly class AggregateInfo
{
    /**
     * @param array<string, string[]> $handlers
     */
    private function __construct(
        public string $type,
        public array $names,
        public array $handlers,
    ) {
    }

    public static function parse(string $type): self
    {
        if (!class_exists($type) || !is_a($type, EventSourceAggregate::class, true)) {
            throw new InvalidArgumentException();
        }

        $reflection = new ReflectionClass($type);

        return new self(
            $type,
            self::resolveNames($reflection),
            self::resolveHandlers($reflection),
        );
    }

    /**
     * @return list<non-empty-string>
     */
    public function handlersByEventType(string $type): array
    {
        return $this->handlers[$type] ?? [];
    }

    /**
     * @param ReflectionClass<EventSourceAggregate> $reflection
     *
     * @return array<string, string[]>
     */
    private static function resolveHandlers(ReflectionClass $reflection): array
    {
        $handlers = [];

        foreach ($reflection->getAttributes(AggregateEmit::class) as $attribute) {
            foreach ($attribute->newInstance()->types as $value) {
                $handlers[$value] = [];
            }
        }

        foreach ($reflection->getMethods() as $method) {
            $attributes = $method->getAttributes(AggregateEventHandler::class);

            if (count($attributes) === 0) {
                continue;
            }

            $parameters = $method->getParameters();

            if (count($parameters) !== 1) {
                throw new LogicException('Aggregate event handler must have exactly one parameter');
            }

            $parameterType = $parameters[0]->getType();

            if (!$parameterType instanceof ReflectionNamedType) {
                throw new LogicException('Aggregate event handler must take an object as a parameter');
            }

            $eventType = $parameterType->getName();

            if (!class_exists($eventType)) {
                throw new LogicException('Aggregate event handler must take an object as a parameter');
            }

            $handlers[$eventType][] = $method->getName();
        }

        return $handlers;
    }

    /**
     * @param ReflectionClass<EventSourceAggregate> $reflection
     *
     * @return string[]
     */
    private static function resolveNames(ReflectionClass $reflection): array
    {
        $names = [];

        foreach ($reflection->getAttributes(AggregateName::class) as $attribute) {
            $names[] = $attribute->newInstance()->value;
        }

        return array_unique($names);
    }
}
