<?php

declare(strict_types=1);

namespace Insight\Component\EventSourcing;

use Insight\Component\EventSourcing\Attribute\EventHandler;
use Insight\Component\EventSourcing\Exception\DefinitionException;
use InvalidArgumentException;
use ReflectionClass;
use ReflectionNamedType;

final class EventHandlersList
{
    /**
     * @var array<class-string, EventHandlersList[]>
     */
    private static array $cache = [];

    /**
     * @param array<class-string, string[]> $handlers
     */
    private function __construct(private readonly array $handlers)
    {
    }

    /**
     */
    public static function fromClass(string $class): self
    {
        return self::$cache[$class] ??= self::parseClass($class);
    }

    /**
     * @return list<string>
     */
    public function forAggregateEvent(object|string $aggregateEvent): array
    {
        if (is_object($aggregateEvent)) {
            $aggregateEvent = get_class($aggregateEvent);
        }

        return $this->handlers[$aggregateEvent] ?? [];
    }

    private static function parseClass(string $class): self
    {
        static $definitionException = new DefinitionException('Event handler must take a valid aggregate event as a first parameter');

        if (!class_exists($class)) {
            throw new InvalidArgumentException("Class $class does not exist");
        }

        $handlers = [];

        $reflection = new ReflectionClass($class);

        foreach ($reflection->getMethods() as $method) {
            if (count($method->getAttributes(EventHandler::class)) === 0) {
                continue;
            }

            $parameters = $method->getParameters();

            if (count($parameters) <= 0) {
                throw $definitionException;
            }

            $type = $parameters[0]->getType();

            if (!$type instanceof ReflectionNamedType) {
                throw $definitionException;
            }

            try {
                $eventInfo = Metadata::get()->aggregateEvent($type->getName());
            } catch (InvalidArgumentException) {
                throw $definitionException;
            }

            $handlers[$eventInfo->type][] = $method->getName();
        }

        return new self($handlers);
    }
}
