<?php

declare(strict_types=1);

namespace Insight\Component\EventSourcing\Mongo;

use Doctrine\ODM\MongoDB\DocumentManager;
use Doctrine\ODM\MongoDB\MongoDBException;
use Insight\Component\EventSourcing\IndexStore;
use Insight\Component\EventSourcing\Mongo\Document\Index;

/**
 * @internal
 *
 * @phpstan-import-type IndexHash from IndexStore
 */
final readonly class MongoIndexStore implements IndexStore
{
    public function __construct(private DocumentManager $documentManager)
    {
    }

    /**
     * {@inheritDoc}
     *
     * @throws MongoDBException
     */
    public function findIndex(string $lookupId, string $name): ?array
    {
        // @phpstan-ignore method.nonObject
        $result = $this->documentManager
            ->createQueryBuilder(Index::class)
                ->field('lookupId')->equals($lookupId)
                ->field('name')->equals($name)
                ->select('hash')
                ->limit(1)
            ->getQuery()
            ->execute()
            ->current();

        return $result instanceof Index
             ? $result->hash
             : null;
    }

    /**
     * {@inheritDoc}
     *
     * @throws MongoDBException
     */
    public function listIndexes(string $lookupId): array
    {
        // @phpstan-ignore method.nonObject
        $result = $this->documentManager
            ->createQueryBuilder(Index::class)
                ->field('lookupId')->equals($lookupId)
                ->select('name')
                ->select('hash')
            ->getQuery()
            ->execute()
            ->toArray();

        $hashes = [];

        /** @var list<Index> $result */
        foreach ($result as $index) {
            $hashes[$index->name] = $index->hash;
        }

        return $hashes;
    }

    /**
     * {@inheritDoc}
     *
     * @throws MongoDBException
     */
    public function lookup(string $name, array $hashTable): array
    {
        // @phpstan-ignore method.nonObject
        $result = $this->documentManager
            ->createQueryBuilder(Index::class)
                ->field('name')->equals($name)
                ->field('hash')->equals($hashTable)
                ->select('lookupId')
            ->getQuery()
            ->execute()
            ->toArray();

        /** @var list<array{lookupId: string}> $result */
        return array_column($result, 'lookupId');
    }

    /**
     * {@inheritDoc}
     *
     * @throws MongoDBException
     */
    public function lookupOne(string $name, array $hashTable): ?string
    {
        // @phpstan-ignore method.nonObject
        $result = $this->documentManager
            ->createQueryBuilder(Index::class)
                ->field('name')->equals($name)
                ->field('hash')->equals($hashTable)
                ->select('lookupId')
                ->limit(1)
            ->getQuery()
            ->execute()
            ->current();

        return $result instanceof Index
             ? $result->lookupId
             : null;
    }

    /**
     * {@inheritDoc}
     *
     * @throws MongoDBException
     */
    public function purge(): void
    {
        $this->documentManager
            ->createQueryBuilder(Index::class)
            ->remove()
            ->getQuery()
            ->execute();
    }

    /**
     * {@inheritDoc}
     */
    public function set(string $lookupId, string $name, array $hashTable): void
    {
        $this->documentManager->persist(
            new Index(
                lookupId: $lookupId,
                name: $name,
                hash: $hashTable,
            ),
        );
    }

    /**
     * {@inheritDoc}
     *
     * @throws MongoDBException
     */
    public function unset(string $lookupId, string $name): void
    {
        $this->documentManager
            ->createQueryBuilder()
            ->remove(Index::class)
                ->field('lookupId')->equals($lookupId)
                ->field('name')->equals($name)
            ->getQuery()
            ->execute();
    }

    /**
     * {@inheritDoc}
     *
     * @throws MongoDBException
     */
    public function unsetLookup(string $lookupId): void
    {
        $this->documentManager
            ->createQueryBuilder()
            ->remove(Index::class)
                ->field('lookupId')->equals($lookupId)
            ->getQuery()
            ->execute();
    }
}
