<?php

declare(strict_types=1);

namespace Insight\Component\EventSourcing\Mongo\Document;

use DateTimeImmutable;
use Doctrine\ODM\MongoDB\Mapping\ClassMetadata;
use Doctrine\Persistence\Mapping\MappingException;
use Insight\Component\EventSourcing\Event as EventInterface;
use Insight\Component\EventSourcing\Exception\UnserializableEventException;
use Insight\Component\EventSourcing\Mongo\Value\Time;
use Throwable;

/**
 * @final
 * @internal
 * @uses self::loadMetadata()
 */
class Event implements EventInterface
{
    /**
     * @param mixed[] $meta
     * @param mixed[] $data
     *
     * @noinspection PhpGetterAndSetterCanBeReplacedWithPropertyHooksInspection
     * @noinspection PhpPropertyCanBeReadonlyInspection
     */
    public function __construct(
        private string $id,
        private int $time,
        private string $type,
        private int $version,
        private string $streamId,
        private int $streamRevision,
        private int $systemRevision,
        private array  $meta,
        private array  $data,
    ) {
    }

    public function as(string $type): object
    {
        try {
            return new $type(...$this->data);
        } catch (Throwable $error) {
            throw new UnserializableEventException($this, $type, $error);
        }
    }

    public function data(): array
    {
        return $this->data;
    }

    public function guid(): string
    {
        return $this->id;
    }

    public function meta(): array
    {
        return $this->meta;
    }

    public function streamId(): string
    {
        return $this->streamId;
    }

    public function streamPosition(): int
    {
        return $this->streamRevision;
    }

    public function systemPosition(): int
    {
        return $this->systemRevision;
    }

    public function time(): DateTimeImmutable
    {
        return Time::fromMilliseconds($this->time)->dateTime;
    }

    public function type(): string
    {
        return $this->type;
    }

    public function version(): int
    {
        return $this->version;
    }

    /**
     * @param ClassMetadata<$this> $metadata
     *
     * @throws MappingException
     *
     * @link https://www.doctrine-project.org/projects/doctrine-mongodb-odm/en/2.9/reference/metadata-drivers.html#core-metadata-drivers
     */
    public static function loadMetadata(ClassMetadata $metadata): void
    {
        $metadata->setCollection('stream');
        $metadata->markReadOnly();

        $metadata->mapField([
            'fieldName' => 'id',
            'type' => 'string',
            'id' => true,
            'strategy' => 'none',
        ]);
        $metadata->mapField([
            'fieldName' => 'time',
            'name' => 'Time',
            'type' => 'integer',
        ]);
        $metadata->mapField([
            'fieldName' => 'type',
            'name' => 'Type',
            'type' => 'string',
        ]);
        $metadata->mapField([
            'fieldName' => 'version',
            'name' => 'Version',
            'type' => 'int',
        ]);
        $metadata->mapField([
            'fieldName' => 'streamId',
            'name' => 'StreamId',
            'type' => 'string',
        ]);
        $metadata->mapField([
            'fieldName' => 'streamRevision',
            'name' => 'StreamRevision',
            'type' => 'integer',
        ]);
        $metadata->mapField([
            'fieldName' => 'systemRevision',
            'name' => 'SystemRevision',
            'type' => 'integer',
        ]);
        $metadata->mapField([
            'fieldName' => 'meta',
            'name' => 'Meta',
            'type' => 'hash',
        ]);
        $metadata->mapField([
            'fieldName' => 'data',
            'name' => 'Data',
            'type' => 'hash',
        ]);

        $metadata->addIndex(['StreamId' => 1]);
        $metadata->addIndex(['StreamId' => 1, 'StreamRevision' =>  1], ['unique' => true]);
        $metadata->addIndex(['StreamId' => 1, 'StreamRevision' => -1], ['unique' => true]);
        $metadata->addIndex(['SystemRevision' =>  1], ['unique' => true]);
        $metadata->addIndex(['SystemRevision' => -1], ['unique' => true]);
        $metadata->addIndex(['Time' =>  1]);
        $metadata->addIndex(['Time' => -1]);
    }
}
