<?php

declare(strict_types=1);

namespace Insight\Component\EventSourcing\Mongo\Document;

use Doctrine\ODM\MongoDB\Mapping\ClassMetadata;
use Doctrine\ODM\MongoDB\Mapping\MappingException;
use Insight\Component\EventSourcing\IndexStore;
use InvalidArgumentException;

/**
 * @final
 * @internal
 * @uses self::loadMetadata()
 *
 * @phpstan-import-type IndexHash from IndexStore
 *
 * @property non-empty-string $name
 * @property IndexHash $hash
 */
class Index
{
    public string $id;

    /**
     * @param mixed[] $hash
     */
    public function __construct(
        public string $lookupId,
        public string $name,
        public  array $hash,
    ) {
        if (strlen($name) === 0) {
            throw new InvalidArgumentException('Index name must be a non-empty string');
        }

        if (count($hash) === 0) {
            throw new InvalidArgumentException('Index hash table must not be empty');
        }

        foreach ($hash as $key => $value) {
            if (!is_string($key) || strlen($key) === 0) {
                throw new InvalidArgumentException('Index hash key must be a non-empty string');
            }
            if (!is_string($value) || strlen($value) === 0) {
                throw new InvalidArgumentException('Index hash value must be a non-empty string');
            }
        }
    }

    /**
     * @param ClassMetadata<$this> $metadata
     *
     * @throws MappingException
     *
     * @link https://www.doctrine-project.org/projects/doctrine-mongodb-odm/en/2.9/reference/metadata-drivers.html#core-metadata-drivers
     */
    public static function loadMetadata(ClassMetadata $metadata): void
    {
        $metadata->setCollection('indexes');
        $metadata->markReadOnly();

        $metadata->mapField([
            'fieldName' => 'id',
            'type' => 'string',
            'id' => true,
            'strategy' => 'auto',
        ]);
        $metadata->mapField([
            'fieldName' => 'lookupId',
            'name' => 'l',
            'type' => 'string',
        ]);
        $metadata->mapField([
            'fieldName' => 'name',
            'name' => 'n',
            'type' => 'string',
        ]);
        $metadata->mapField([
            'fieldName' => 'hash',
            'name' => 'h',
            'type' => 'hash',
        ]);

        $metadata->addIndex(
            ['l' => 1, 'n' => 1, 'h' => 1],
            ['unique' => true],
        );
    }
}
