<?php

declare(strict_types=1);

namespace Insight\Component\EventSourcing;

/**
 * @phpstan-type IndexHash non-empty-array<non-empty-string, non-empty-string>
 */
interface IndexStore
{
    /**
     * Searches for an index associated with the specific lookup. Returns NULL if no such index exist.
     *
     * @return IndexHash|null
     */
    public function findIndex(string $lookupId, string $name): ?array;

    /**
     * Searches for indexes associated with the specific lookup.
     *
     * @return array<string, IndexHash>
     */
    public function listIndexes(string $lookupId): array;

    /**
     * Returns all lookup ids matching given index.
     *
     * @param IndexHash $hashTable
     *
     * @return list<string>
     */
    public function lookup(string $name, array $hashTable): array;

    /**
     * Returns one lookup id matching given index or NULL if such does not exist.
     *
     * @param IndexHash $hashTable
     */
    public function lookupOne(string $name, array $hashTable): ?string;

    /**
     * Removes all indexes from the store.
     */
    public function purge(): void;

    /**
     * Sets new index.
     *
     * This method SHOULD pass silently if index already exists, as it should be up to the event store to handle that case.
     *
     * @param IndexHash $hashTable
     */
    public function set(string $lookupId, string $name, array $hashTable): void;

    /**
     * Removes (releases) an index.
     */
    public function unset(string $lookupId, string $name): void;

    /**
     * Removes (releases) all indexes for a given lookup.
     */
    public function unsetLookup(string $lookupId): void;
}
