<?php

declare(strict_types=1);

namespace Insight\Component\EventSourcing\Mongo;

use Doctrine\ODM\MongoDB\DocumentManager;
use Doctrine\ODM\MongoDB\MongoDBException;
use Insight\Component\EventSourcing\IndexStore;
use Insight\Component\EventSourcing\Mongo\Document\Index;

/**
 * @internal
 */
final readonly class MongoIndexStore implements IndexStore
{
    public function __construct(private DocumentManager $documentManager)
    {
    }

    /**
     * {@inheritDoc}
     *
     * @throws MongoDBException
     */
    public function lookup(string $namespace, string $key, string $value): array
    {
        return $this->documentManager
            ->getDocumentCollection(Index::class)
            ->distinct('l', [
                'n' => $namespace,
                'k' => $key,
                'v' => $value,
            ]);
    }

    /**
     * {@inheritDoc}
     *
     * @throws MongoDBException
     */
    public function lookupOne(string $namespace, string $key, string $value): ?string
    {
        $lookup = $this->lookup($namespace, $key, $value);

        return match (count($lookup)) {
            0 => null,
            1 => $lookup[0],
            default => throw new \Exception('TOO MANY RESULTS'),
        };
    }

    /**
     * {@inheritDoc}
     *
     * @throws MongoDBException
     */
    public function purge(): void
    {
        $this->documentManager
            ->createQueryBuilder(Index::class)
            ->remove()
            ->getQuery()
            ->execute();
    }

    /**
     * {@inheritDoc}
     */
    public function set(string $lookupId, string $namespace, string $key, string $value): void
    {
        $this->documentManager->persist(
            new Index($namespace, $lookupId, $key, $value),
        );
    }

    /**
     * {@inheritDoc}
     *
     * @throws MongoDBException
     */
    public function unset(string $lookupId, string $namespace, string $key, string $value): void
    {
        $this->documentManager
            ->createQueryBuilder()
            ->remove(Index::class)
                ->field('n')->equals($namespace)
                ->field('l')->equals($lookupId)
                ->field('k')->equals($key)
                ->field('v')->equals($value)
            ->getQuery()
            ->execute();
    }

    /**
     * {@inheritDoc}
     *
     * @throws MongoDBException
     */
    public function unsetLookup(string $lookupId): void
    {
        $this->documentManager
            ->createQueryBuilder()
            ->remove(Index::class)
                ->field('l')->equals($lookupId)
            ->getQuery()
            ->execute();
    }
}
