<?php

declare(strict_types=1);

namespace Insight\Component\EventSourcing\Mongo;

use DateTimeImmutable;
use Doctrine\ODM\MongoDB\DocumentManager;
use Doctrine\ODM\MongoDB\MongoDBException;
use Generator;
use Insight\Component\EventSourcing\EventStore;
use Insight\Component\EventSourcing\Exception\UnexpectedRevisionException;
use Insight\Component\EventSourcing\Exception\UniqueConstraintException;
use Insight\Component\EventSourcing\Exception\UnknownIndexException;
use Insight\Component\EventSourcing\Index;
use Insight\Component\EventSourcing\Mongo\Doctrine\CustomDriver;
use Insight\Component\EventSourcing\Mongo\Document\Event;
use Insight\Component\EventSourcing\Mongo\Document\Index as IndexDocument;
use Insight\Component\EventSourcing\Mongo\Document\Position;
use Insight\Component\EventSourcing\Mongo\Document\System;
use Insight\Component\EventSourcing\Mongo\Value\Guid;
use Insight\Component\EventSourcing\Mongo\Value\Time;
use Insight\Component\EventSourcing\Record;
use InvalidArgumentException;
use LogicException;
use MongoDB\Driver\Exception\BulkWriteException;
use Throwable;

/**
 * @template-implements EventStore<Event>
 */
final class MongoEventStore implements EventStore
{
    use CustomDriver;

    /**
     * @var list<Index>
     */
    private array $indexes = [];

    private MongoIndexStore $indexStore;

    public function __construct(private readonly DocumentManager $documentManager)
    {
        $this->loadMetadataDriver($this->documentManager);

        $this->indexStore = new MongoIndexStore($this->documentManager);
    }

    public function addIndex(Index $index): void
    {
        $this->indexes[$index::class] = $index->withIndexStore($this->indexStore);
    }

    public function getIndex(string $class): Index
    {
        $index = $this->indexes[$class] ?? throw new UnknownIndexException($class);

        return $index->withIndexStore($this->indexStore);
    }

    /**
     * @throws MongoDBException
     */
    public function append(string $streamId, int $streamRevision, array $records): void
    {
        $currentRevision = $this->stream($streamId)->currentRevision();

        if ($streamRevision !== $currentRevision) {
            throw new UnexpectedRevisionException($streamId, $streamRevision, $currentRevision);
        }

        foreach ($records as $record) {
            $record instanceof Record || throw new InvalidArgumentException();

            $event = new Event(
                id: Guid::generate()->toString(),
                time: Time::now()->milliseconds(),
                type: $record->type,
                version: $record->version,
                streamId: $streamId,
                streamRevision: ++$streamRevision,
                systemRevision: $this->reserveSystemVersion(),
                meta: $record->meta,
                data: $record->data,
            );

            $this->documentManager->persist($event);

            foreach ($this->indexes as $index) {
                $index->when($event);
            }
        }
    }

    /**
     * @throws Throwable
     */
    public function commit(): void
    {
        try {
            $this->documentManager->flush();
        } catch (BulkWriteException $exception) {
            if ($exception->getCode() === 11000) {
                throw new UniqueConstraintException(previous: $exception);
            }
            throw $exception;
        } finally {
            $this->documentManager->clear();
        }
    }

    public function initialize(): void
    {
        $schemaManager = $this->documentManager->getSchemaManager();

        $schemaManager->ensureDocumentIndexes(Event::class);
        $schemaManager->ensureDocumentIndexes(IndexDocument::class);
        $schemaManager->ensureDocumentIndexes(Position::class);
        $schemaManager->ensureDocumentIndexes(System::class);
    }

    public function listen(?int $fromRevision = null): Generator
    {
        return $this->streamAll()->listen($fromRevision);
    }

    /**
     * @throws Throwable
     */
    public function refreshIndexes(): void
    {
        $this->indexStore->purge();

        foreach ($this->streamAll()->all() as $i => $event) {
            foreach ($this->indexes as $index) {
                $index->when($event);
            }

            if ($i % 100 === 0) {
                $this->commit();
            }
        }

        $this->commit();
    }

    public function stream(string $streamId): MongoEventStream
    {
        return new MongoEventStream($this->documentManager, $streamId);
    }

    public function streamAll(): MongoEventStream
    {
        return new MongoEventStream($this->documentManager);
    }

    /**
     * @throws MongoDBException
     */
    private function reserveSystemVersion(): int
    {
        $systemInfo = $this->documentManager
            ->getRepository(System::class)
            ->createQueryBuilder()
                ->findAndUpdate()
                ->returnNew()
                ->upsert()
                ->field('_id')->equals('System')
                ->field('revision')->inc(1)
                ->field('createdAt')->setOnInsert(new DateTimeImmutable())
                ->field('updatedAt')->set(new DateTimeImmutable())
            ->getQuery()
            ->execute();

        if (!$systemInfo instanceof System) {
            throw new LogicException();
        }

        return $systemInfo->revision;
    }
}
