<?php

declare(strict_types=1);

namespace Insight\Component\EventSourcing;

final readonly class Projector
{
    public function __construct(private EventStore $store)
    {
    }

    /**
     * @param iterable<int, Projection> $projections
     */
    public function start(Projection ...$projections): void
    {
        if (count($projections) === 0) {
            return;
        }

        $lowestProjectionRevision = $this->findLowestRevision($projections);

        $events = $this->store->streamAll()->listen(fromRevision: $lowestProjectionRevision);

        foreach ($events as $event) {
            array_walk(
                $projections,
                static fn (Projection $projection) => $projection->when($event),
            );
        }
    }

    /**
     * @param iterable<int, Projection> $projections
     */
    public function synchronize(Projection ...$projections): void
    {
        if (count($projections) === 0) {
            return;
        }

        $lowestProjectionRevision = $this->findLowestRevision($projections);

        $events = $this->store->streamAll()->above(revision: $lowestProjectionRevision);

        foreach ($events as $event) {
            array_walk(
                $projections,
                static fn (Projection $projection) => $projection->when($event),
            );
        }
    }

    /**
     * @param non-empty-array<Projection> $projections
     */
    private function findLowestRevision(array $projections): int
    {
        return min(array_map(
            static fn (Projection $projection) => $projection->revision(),
            $projections,
        ));
    }
}
