<?php

declare(strict_types=1);

namespace Insight\Component\EventSourcing\Mongo;

use Doctrine\ODM\MongoDB\DocumentManager;
use Doctrine\ODM\MongoDB\Query\Builder;
use Doctrine\Persistence\ObjectRepository;
use Insight\Component\EventSourcing\Event;
use Insight\Component\EventSourcing\EventStore;
use Insight\Component\EventSourcing\Mongo\Doctrine\CustomDriver;
use Insight\Component\EventSourcing\Mongo\Document\Position;
use Insight\Component\EventSourcing\Mongo\Exception\DependencyException;
use Insight\Component\EventSourcing\Projection;
use Insight\Component\EventSourcing\Projection\WithHandlers;
use InvalidArgumentException;
use LogicException;
use Throwable;

DependencyException::check();

/**
 * @template TDocument
 */
abstract class MongoProjection implements Projection
{
    use CustomDriver;
    use WithHandlers;

    /**
     * @var class-string<TDocument>
     */
    protected static string $class;

    /**
     * @var Position<TDocument>
     */
    private Position $position;

    final public function __construct(protected readonly DocumentManager $documentManager)
    {
        $this->loadMetadataDriver($this->documentManager);

        try {
            $this->documentManager->getClassMetadata(static::$class);
        } catch (Throwable) {
            throw new LogicException(
                sprintf("Document class is either not set or not properly mapped (in %s)", static::class),
            );
        }

        $this->position = $this->position();
    }

    /**
     * @throws Throwable
     */
    final public function sync(EventStore $store): void
    {
        foreach ($store->streamAll()->above($this->revision()) as $event) {
            $this->when($event);
        }
    }

    /**
     * @throws Throwable
     */
    final public function when(Event $event): void
    {
        if ($this->hasBeenProjected($event)) {
            return;
        }

        $this->handle($event);

        $this->position->revision = $event->systemPosition();

        $this->documentManager->flush();
    }

    /**
     * @throws Throwable
     */
    final public function reset(): void
    {
        $this
            ->createQueryBuilder()
            ->remove()
            ->getQuery()
            ->execute();

        $this->documentManager->remove($this->position);
        $this->documentManager->flush();

        $this->position = $this->position();
    }

    final public function revision(): int
    {
        return $this->position->revision;
    }

    final protected function createQueryBuilder(): Builder
    {
        return $this->getRepository()->createQueryBuilder();
    }

    /**
     * @return ObjectRepository<TDocument>
     */
    final protected function getRepository(): ObjectRepository
    {
        return $this->documentManager->getRepository(static::$class);
    }

    /**
     * @return TDocument|null
     */
    final protected function find(string $id)
    {
        return $this->getRepository()->find($id);
    }

    /**
     * @return TDocument|null
     */
    final protected function findOneBy(array $criteria)
    {
        return $this->getRepository()->findOneBy($criteria);
    }

    /**
     * @param TDocument $document
     */
    final protected function remove($document): void
    {
        if (!is_a($document, static::$class, true)) {
            throw new InvalidArgumentException('Attempted to save an invalid document type');
        }

        $this->documentManager->remove($document);
    }

    final protected function removeById(string $id): void
    {
        $this->documentManager->remove(
            $this->documentManager->getReference(static::$class, $id),
        );
    }

    /**
     * @param TDocument $document
     *
     * @return TDocument
     */
    final protected function save($document)
    {
        if (!is_a($document, static::$class, true)) {
            throw new InvalidArgumentException('Attempted to save an invalid document type');
        }

        $this->documentManager->persist($document);

        return $document;
    }

    private function hasBeenProjected(Event $event): bool
    {
        return $this->revision() >= $event->systemPosition();
    }

    private function position(): Position
    {
        $position = $this->documentManager->find(Position::class, static::$class);

        if (null === $position) {
            $this->documentManager->persist(
                $position = new Position(static::$class, 0),
            );
        }

        return $position;
    }
}
