<?php

declare(strict_types=1);

namespace Insight\Component\EventSourcing\Mongo;

use Doctrine\ODM\MongoDB\DocumentManager;
use Doctrine\ODM\MongoDB\MongoDBException;
use Insight\Component\EventSourcing\IndexStore;
use Insight\Component\EventSourcing\Mongo\Document\Index;
use Throwable;

/**
 * @internal
 */
final readonly class MongoIndexStore implements IndexStore
{
    public function __construct(private DocumentManager $documentManager)
    {
    }

    /**
     * {@inheritDoc}
     *
     * @throws MongoDBException
     */
    public function exists(string $index): bool
    {
        return $this->documentManager->getDocumentCollection(Index::class)->countDocuments(['name' => $index]) > 0;
    }

    /**
     * {@inheritDoc}
     *
     * @throws Throwable
     */
    public function set(string $index, array $tags = []): void
    {
        $indexes[] = $index;

        foreach ($indexes as $i) {
            $this->documentManager->persist(new Index($i, $tags));
        }
    }

    /**
     * {@inheritDoc}
     */
    public function tagged(array $tags): array
    {
        $result = $this->documentManager
            ->getRepository(Index::class)
            ->createQueryBuilder()
                ->select(['index'])
                ->field('tags')->all($tags)
            ->getQuery()
            ->toArray();

        return array_map(
            static fn (Index $index) => $index->index,
            $result,
        );
    }

    /**
     * {@inheritDoc}
     */
    public function unset(string $index): void
    {
        $indexes[] = $index;

        foreach ($indexes as $i) {
            $this->documentManager->remove(
                $this->documentManager->getReference(Index::class, $i),
            );
        }
    }

    /**
     * {@inheritDoc}
     *
     * @throws MongoDBException
     */
    public function unsetTagged(array $tags): void
    {
        if (count($tags) === 0) {
            return;
        }

        $this->documentManager
            ->getRepository(Index::class)
            ->createQueryBuilder()
                ->remove()
                ->field('tags')->all($tags)
            ->getQuery()
            ->execute();
    }
}
