<?php

declare(strict_types=1);

namespace Insight\Bundle\EventSourcingBundle;

use Insight\Bundle\EventSourcingBundle\Exception\InvalidConfigurationException;
use Insight\Bundle\EventSourcingBundle\Extension\Store\MongoStoreExtension;
use Insight\Bundle\EventSourcingBundle\Extension\StoreExtension;
use Insight\Bundle\EventSourcingBundle\Projector\ProjectorWorker;
use Insight\Component\EventSourcing\EventSourceAggregateRepository;
use Insight\Component\EventSourcing\EventStore;
use Insight\Component\EventSourcing\Metadata;
use Insight\Component\EventSourcing\Projector;
use Symfony\Component\Config\Definition\Configurator\DefinitionConfigurator;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Exception\ServiceNotFoundException;
use Symfony\Component\DependencyInjection\Loader\Configurator\ContainerConfigurator;
use Symfony\Component\DependencyInjection\Reference;
use Symfony\Component\HttpKernel\Bundle\AbstractBundle;

final class EventSourcingBundle extends AbstractBundle
{
    public function boot(): void
    {
        $metadata = Metadata::get();

        // @todo cache
        foreach ($this->container->getParameter('es.aggregates.preload') as $aggregateType) {
            $metadata->aggregate($aggregateType);
        }

        parent::boot();
    }

    public function configure(DefinitionConfigurator $definition): void
    {
        $definition->import('../config/definitions/*.php');
    }

    public function loadExtension(array $config, ContainerConfigurator $container, ContainerBuilder $builder): void
    {
        $container->import('../config/services.php');

        $this->loadExtensions($config, $builder);
        $this->loadAggregates($config, $container);
        $this->loadStores($config['stores'], $container, $builder);
        $this->loadProjectors($config['projectors'], $container);
    }

    private function loadAggregates(array $config, ContainerConfigurator $container): void
    {
        $container->parameters()->set('es.aggregates.preload', $config['aggregates']);
    }

    private function loadExtensions(array $config, ContainerBuilder $builder): void
    {
        $classes = array_merge(
            [
                MongoStoreExtension::class,
            ],
            $config['extensions'] ?? []
        );

        foreach ($classes as $class) {
            if (!class_exists($class) || !is_a($class, StoreExtension::class, true)) {
                throw new InvalidConfigurationException("$class is not a valid store extension");
            }

            $extension = new $class();

            $extensions[$extension->type()] = $extension;
        }

        $builder->setParameter('es.extensions', $extensions);
    }

    private function loadProjector(string $workerId, array $workerConfig, ContainerConfigurator $container): void
    {
        ['store' => $storeId, 'projections' => $projections] = $workerConfig;

        $services = $container->services();

        try {
            $services->get($projectorId = "es.projector.$storeId");
        } catch (ServiceNotFoundException) {
            throw new InvalidConfigurationException("Store `$storeId` for projector worker `$workerId` does not exist.");
        }

        $projections = array_map(
            static fn (string $projectionServiceId) => new Reference($projectionServiceId),
            $projections,
        );

        $services
            ->set("es.worker.$workerId", ProjectorWorker::class)
            ->tag('es.worker', ['id' => $workerId])
            ->args([
                new Reference($projectorId),
                $projections,
            ]);
    }

    private function loadProjectors(array $config, ContainerConfigurator $container): void
    {
        foreach ($config as $workerId => $workerConfig) {
            $this->loadProjector((string) $workerId, $workerConfig, $container);
        }
    }

    private function loadStores(array $config, ContainerConfigurator $container, ContainerBuilder $builder): void
    {
        $extensions = $builder->getParameter('es.extensions');

        $services = $container->services();

        $stores = [];

        foreach ($config as $storeId => $storeConfig) {
            $type = $storeConfig['type'];

            $extension = $extensions[$type] ?? null;

            if (!$extension instanceof StoreExtension) {
                throw new InvalidConfigurationException("There is no store extension that supports type: $type");
            }

            $storeDefinition = $extension->registerStore($builder, $storeConfig['options'] ?? []);
            $storeDefinition->addTag('es.store', ['id' => $storeId]);
            $builder->setDefinition("es.store.$storeId", $storeDefinition);

            foreach ($storeConfig['indexes'] as $indexService) {
                $storeDefinition->addMethodCall('addIndex', [new Reference($indexService)]);
            }

            $services
                ->set("es.projector.$storeId", Projector::class)
                ->tag('es.projector', ['id' => $storeId])
                ->args([
                    '$store' => new Reference("es.store.$storeId"),
                ]);

            $services
                ->set("es.repository.$storeId", EventSourceAggregateRepository::class)
                ->tag('es.repository', ['id' => $storeId])
                ->args([
                    '$store' => new Reference("es.store.$storeId"),
                ]);

            $stores[] = $storeId;
        }

        if (count($stores) === 1) {
            $defaultId = reset($stores);
            $container
                ->services()
                ->alias(EventStore::class, "es.store.$defaultId")
                ->alias(Projector::class, "es.projector.$defaultId")
                ->alias(EventSourceAggregateRepository::class, "es.repository.$defaultId");
        }

        $builder->setParameter('es.extensions', null);
    }
}
