<?php

declare(strict_types=1);

namespace Tests\DawidRza\Component\Persistence;

use DawidRza\Component\Persistence\MongoDB\Exception\MappingException;
use DawidRza\Component\Persistence\MongoDB\MongoRepository;
use stdClass;
use Tests\DawidRza\Component\Persistence\_Type\ExampleDocument;

final class ProjectTest extends BasicTestCase
{
    public function testCheckWhetherObjectsWithoutMetadataAreRejected(): void
    {
        $this->expectException(MappingException::class);
        new class($this->getUnitOfWork()) extends MongoRepository {
            protected static function getDocumentClassName(): string
            {
                return stdClass::class;
            }
        };
    }

    public static function testCheckWhetherRepositoryDoesntFindNonExistingElements(): void
    {
        $documentId = self::getUniqueId();

        $repository = self::getRepository();
        self::assertNull($repository->_findDocumentById($documentId));
    }

    public static function testCheckWhetherInsertingAndNotApplyingChangesDoesntChangeState(): void
    {
        $repository = self::getRepository();

        $documentId = self::getUniqueId();
        self::assertNull($repository->_findDocumentById($documentId));

        $documentToSave = new ExampleDocument($documentId, 'test-name');
        $repository->_scheduleDocumentForUpsert($documentToSave);

        self::assertNotNull($repository->_findDocumentById($documentId));
    }

    public static function testCheckWhetherInsertingAndClearingDoesntChangeState(): void
    {
        $unitOfWork = self::getUnitOfWork();
        $repository = self::getRepository();

        $documentId = self::getUniqueId();
        self::assertNull($repository->_findDocumentById($documentId));

        $documentToSave = new ExampleDocument($documentId, 'test-name');
        $repository->_scheduleDocumentForUpsert($documentToSave);
        $unitOfWork->clear();

        self::assertNull($repository->_findDocumentById($documentId));
    }

    public static function testCheckWhetherInsertingAndApplyingChangesChangesState(): void
    {
        $unitOfWork = self::getUnitOfWork();
        $repository = self::getRepository();

        $documentId = self::getUniqueId();
        self::assertNull($repository->_findDocumentById($documentId));

        $documentToSave = new ExampleDocument($documentId, 'test-name');
        $repository->_scheduleDocumentForUpsert($documentToSave);
        $unitOfWork->applyChanges();

        self::assertNotNull($repository->_findDocumentById($documentId));
    }

    public static function testCheckWhetherUpdatingAndNotApplyingChangesDoesntUpdateState(): void
    {
        $unitOfWork = self::getUnitOfWork();
        $repository = self::getRepository();

        $documentId = self::getUniqueId();
        self::assertNull($repository->_findDocumentById($documentId));

        $documentToSave = new ExampleDocument($documentId, 'test-name');
        $repository->_scheduleDocumentForUpsert($documentToSave);
        $unitOfWork->applyChanges();
        $documentToSave->name = 'test-name2';

        $foundDocument = $repository->_findDocumentById($documentId);
        self::assertNotNull($foundDocument);
        self::assertSame('test-name2', $foundDocument->name);
    }

    public static function testCheckWhetherUpdatingAndClearingDoesntUpdateState(): void
    {
        $unitOfWork = self::getUnitOfWork();
        $repository = self::getRepository();

        $documentId = self::getUniqueId();
        self::assertNull($repository->_findDocumentById($documentId));

        $documentToSave = new ExampleDocument($documentId, 'test-name');
        $repository->_scheduleDocumentForUpsert($documentToSave);
        $unitOfWork->applyChanges();
        $documentToSave->changeName('test-name2');
        $unitOfWork->clear();

        $foundDocument = $repository->_findDocumentById($documentId);
        self::assertNotNull($foundDocument);
        self::assertSame('test-name', $foundDocument->name);
    }

    public static function testCheckWhetherUpdatingAndApplyingChangesUpdatesState(): void
    {
        $unitOfWork = self::getUnitOfWork();
        $repository = self::getRepository();

        $documentId = self::getUniqueId();
        self::assertNull($repository->_findDocumentById($documentId));

        $documentToSave = new ExampleDocument($documentId, 'test-name');
        $repository->_scheduleDocumentForUpsert($documentToSave);
        $unitOfWork->applyChanges();
        $documentToSave->changeName('test-name2');
        $unitOfWork->applyChanges();

        $foundDocument = $repository->_findDocumentById($documentId);
        self::assertNotNull($foundDocument);
        self::assertSame('test-name2', $foundDocument->name);
    }

    /* TODO: Is this correct behaviour? */
    public static function testCheckWhetherClearingDetachesDocumentsFromRepository(): void
    {
        $unitOfWork = self::getUnitOfWork();
        $repository = self::getRepository();

        $documentId = self::getUniqueId();
        self::assertNull($repository->_findDocumentById($documentId));

        $documentToSave = new ExampleDocument($documentId, 'test-name');
        $repository->_scheduleDocumentForUpsert($documentToSave);
        $unitOfWork->applyChanges();
        $unitOfWork->clear();
        $documentToSave->changeName('test-name2');
        $unitOfWork->applyChanges();

        $foundDocument = $repository->_findDocumentById($documentId);
        self::assertNotNull($foundDocument);
        self::assertSame('test-name', $foundDocument->name);
    }

    public static function testCheckWhetherRemovingNonexistentDocumentDoesntDoAnything(): void
    {
        $unitOfWork = self::getUnitOfWork();
        $repository = self::getRepository();

        $documentId = self::getUniqueId();
        self::assertNull($repository->_findDocumentById($documentId));

        $documentToSave = new ExampleDocument($documentId, 'test-name');
        $repository->_scheduleDocumentForRemove($documentToSave);
        $unitOfWork->applyChanges();

        self::assertNull($repository->_findDocumentById($documentId));
    }

    public static function testCheckWhetherRemovingAndNotApplyingChangesDoesntDoAnything(): void
    {
        $unitOfWork = self::getUnitOfWork();
        $repository = self::getRepository();

        $documentId = self::getUniqueId();
        self::assertNull($repository->_findDocumentById($documentId));

        $documentToSave = new ExampleDocument($documentId, 'test-name');
        $repository->_scheduleDocumentForUpsert($documentToSave);
        $unitOfWork->applyChanges();
        $repository->_scheduleDocumentForRemove($documentToSave);

        self::assertNotNull($repository->_findDocumentById($documentId));
    }

    public static function testCheckWhetherRemovingAndClearingDoesntDoAnything(): void
    {
        $unitOfWork = self::getUnitOfWork();
        $repository = self::getRepository();

        $documentId = self::getUniqueId();
        self::assertNull($repository->_findDocumentById($documentId));

        $documentToSave = new ExampleDocument($documentId, 'test-name');
        $repository->_scheduleDocumentForUpsert($documentToSave);
        $unitOfWork->applyChanges();
        $repository->_scheduleDocumentForRemove($documentToSave);
        $unitOfWork->clear();

        self::assertNotNull($repository->_findDocumentById($documentId));
    }

    public static function testCheckWhetherRemovingAndApplyingChangesRemovesDocument(): void
    {
        $unitOfWork = self::getUnitOfWork();
        $repository = self::getRepository();

        $documentId = self::getUniqueId();
        self::assertNull($repository->_findDocumentById($documentId));

        $documentToSave = new ExampleDocument($documentId, 'test-name');
        $repository->_scheduleDocumentForUpsert($documentToSave);
        $unitOfWork->applyChanges();
        $repository->_scheduleDocumentForRemove($documentToSave);
        $unitOfWork->applyChanges();

        self::assertNull($repository->_findDocumentById($documentId));
    }

    /**
     * @return non-empty-string
     */
    private static function getUniqueId(): string
    {
        return bin2hex(random_bytes(32));
    }
}
