<?php

declare(strict_types=1);

namespace DawidRza\Component\Persistence\MongoDB;

use DawidRza\Component\Persistence\EventPublisher;
use DawidRza\Component\Persistence\UnitOfWork;
use Doctrine\ODM\MongoDB\DocumentManager;
use Doctrine\ODM\MongoDB\Events;
use Doctrine\ODM\MongoDB\MongoDBException;
use LogicException;
use Throwable;
use WeakMap;

/**
 * @internal Only for internal use, not part of the public API.
 */
final readonly class MongoUOW implements UnitOfWork
{
    public DocumentManager $documentManager;

    private ?DoctrineEventListener $doctrineEventListener;

    /**
     * @param ?EventPublisher<object> $eventPublisher
     */
    public function __construct(
        DocumentManager $documentManager,
        ?EventPublisher $eventPublisher = null,
    ) {
        $this->documentManager = $documentManager;

        if ($eventPublisher !== null) {
            $this->doctrineEventListener = $this->registerEventListener($eventPublisher);
        } else {
            $this->doctrineEventListener = null;
        }
    }

    /**
     * @throws MongoDBException
     * @throws Throwable
     */
    public function applyChanges(): void
    {
        $this->documentManager->flush();
        $this->doctrineEventListener?->commit();
    }

    public function clear(): void
    {
        $this->documentManager->clear();
    }

    /**
     * @param EventPublisher<object> $eventPublisher
     */
    private function registerEventListener(EventPublisher $eventPublisher): DoctrineEventListener
    {
        /** @var WeakMap<DocumentManager, DoctrineEventListener> $listeners */
        static $listeners = new WeakMap();

        if ($listeners->offsetExists($this->documentManager)) {
            throw new LogicException('The document manager has already been registered');
        }

        $listener = new DoctrineEventListener(
            eventPublisher: $eventPublisher,
            documentManager: $this->documentManager,
        );

        $this->documentManager->getEventManager()->addEventListener(
            events: [Events::postRemove, Events::postUpdate, Events::postPersist],
            listener: $listener,
        );

        return $listeners[$this->documentManager] = $listener;
    }
}
