<?php

declare(strict_types=1);

namespace DawidRza\Component\Persistence\MongoDB;

use DawidRza\Component\Persistence\EventPublisher;
use DawidRza\Component\Persistence\WithEvents;
use Doctrine\ODM\MongoDB\DocumentManager;
use Doctrine\ODM\MongoDB\Event\LifecycleEventArgs;
use LogicException;
use ReflectionProperty;

final class DoctrineEventListener
{
    /** @var array<class-string, ?ReflectionProperty> $cache */
    private static array $cache = [];

    /** @var list<object> $documents */
    private array $documents = [];

    /**
     * @param EventPublisher<object> $eventPublisher
     */
    public function __construct(
        private readonly EventPublisher $eventPublisher,
        private readonly DocumentManager $documentManager,
    ) {
    }

    public function commit(): void
    {
        $events = [];
        foreach ($this->documents as $document) {
            $property = $this->getEventsProperty($document);
            if ($property === null) {
                continue;
            }

            $documentEvents = $property->getValue($document);
            if (!is_array($documentEvents)) {
                throw new LogicException('Expected the events property in document to be an array');
            }

            /** @var list<object> $documentEvents */
            array_push($events, ...$documentEvents);

            $property->setValue($document, []);
        }

        $this->documents = [];

        $this->eventPublisher->publish(...$events);
    }

    public function postRemove(LifecycleEventArgs $args): void
    {
        $this->registerDocument($args);
    }

    public function postUpdate(LifecycleEventArgs $args): void
    {
        $this->registerDocument($args);
    }

    public function postPersist(LifecycleEventArgs $args): void
    {
        $this->registerDocument($args);
    }

    private function registerDocument(LifecycleEventArgs $args): void
    {
        if ($this->documentManager !== $args->getDocumentManager()) {
            throw new LogicException('Attempted to register a document with mismatched document manager');
        }

        $this->documents[] = $args->getDocument();
    }

    private function getEventsProperty(object $document): ?ReflectionProperty
    {
        if (array_key_exists($document::class, self::$cache)) {
            return self::$cache[$document::class];
        }

        if (!$this->usesTrait($document)) {
            return self::$cache[$document::class] = null;
        } else {
            return self::$cache[$document::class] = new ReflectionProperty($document, 'events');
        }
    }

    private function usesTrait(object $object): bool
    {
        return in_array(WithEvents::class, class_uses($object), true);
    }
}
