<?php

declare(strict_types=1);

namespace DawidRza\Component\Persistence\MongoDB;

use DawidRza\Component\Persistence\MongoDB\Exception\MappingException;
use Doctrine\ODM\MongoDB\DocumentManager;
use Throwable;
use UnexpectedValueException;

/**
 * Simple common parent for all classes that are meant for browsing MongoDB documents.
 */
abstract class MongoReader
{
    public function __construct(
        private readonly DocumentManager $documentManager,
    ) {
    }

    /**
     * Executes count pipeline on the aggregation builder and returns the result.
     *
     * @return non-negative-int
     */
    final protected function count(AggregationBuilder $builder): int
    {
        $aggregationResult = $builder
            ->clone(resultType: null)
            ->count('count')
            ->getAggregation()
            ->getSingleResult();

        if ($aggregationResult === null) {
            return 0;
        }

        if (!is_array($aggregationResult) || !array_key_exists('count', $aggregationResult)) {
            throw new UnexpectedValueException('Expected the aggregation result to include a count');
        }

        $count = $aggregationResult['count'];

        if (!is_int($count) || $count < 0) {
            throw new UnexpectedValueException('Expected the aggregation result to include a count that holds a non-negative integer');
        }

        return $count;
    }

    /**
     * @param class-string $documentType
     * @param class-string $resultType
     */
    final protected function createAggregationBuilder(string $documentType, string $resultType): AggregationBuilder
    {
        $this->verifyDocumentType($documentType);

        $builder = new AggregationBuilder($this->documentManager, $documentType);

        return $builder->hydrate($resultType);
    }

    /**
     * @template R of object
     *
     * @param class-string $documentType
     * @param class-string<R> $resultType
     *
     * @return ?R
     */
    final protected function queryById(string $documentType, mixed $documentId, string $resultType): ?object
    {
        $result = $this
            ->createAggregationBuilder($documentType, $resultType)
            ->match()
                ->field('_id')->equals($documentId)
            ->getAggregation()
            ->getSingleResult();

        if ($result !== null && !$result instanceof $resultType) {
            throw new UnexpectedValueException(sprintf(
                "Critical error: result was expected to be an instance of %s but %s was returned.",
                $resultType,
                get_debug_type($result),
            ));
        }

        return $result;
    }

    /**
     * @param class-string $documentType
     */
    private function verifyDocumentType(string $documentType): void
    {
        /**
         * @var array<class-string, true> $VERIFIED
         */
        static $VERIFIED = [];

        if (array_key_exists($documentType, $VERIFIED)) {
            return;
        }

        try {
            $this->documentManager->getClassMetadata($documentType);
        } catch (Throwable $exception) {
            throw MappingException::documentIsNotSet($documentType, $exception);
        }

        $VERIFIED[$documentType] = true;
    }
}
