<?php

declare(strict_types=1);

namespace DawidRza\Component\Persistence\MongoDB;

use DawidRza\Component\Persistence\MongoDB\Exception\MappingException;
use Doctrine\ODM\MongoDB\Aggregation\Builder as AggregationBuilder;
use Doctrine\ODM\MongoDB\Query\Builder as QueryBuilder;
use LogicException;
use Throwable;

/**
 * @template TRoot of object
 */
abstract class MongoRepository
{
    /**
     * @var non-empty-list<class-string>
     */
    private array $documentClassNameList;

    public function __construct(
        private readonly MongoUOW $uow,
    ) {
        $this->registerDocumentClassNameList();
    }

    /**
     * @return class-string<TRoot>
     */
    abstract protected static function getDocumentClassName(): string;

    /**
     * @param ?class-string $documentClassName
     */
    final protected function createAggregationBuilder(?string $documentClassName = null): AggregationBuilder
    {
        return $this->uow->documentManager->createAggregationBuilder(
            $this->acceptDocumentClassName($documentClassName),
        );
    }

    /**
     * @param ?class-string $documentClassName
     */
    final protected function createQueryBuilder(?string $documentClassName = null): QueryBuilder
    {
        return $this->uow->documentManager->createQueryBuilder(
            $this->acceptDocumentClassName($documentClassName),
        );
    }

    /**
     * @return ?TRoot
     */
    final protected function findDocumentById(mixed $documentId): ?object
    {
        // @TODO: Should we allow finding referenced documents directly or only through the query builder?

        /** @var class-string<TRoot> $documentClassName */
        $documentClassName = $this->documentClassNameList[0];

        return $this->uow->documentManager->find(
            $documentClassName,
            $documentId,
        );
    }

    final protected function scheduleDocumentForRemove(object $document): void
    {
        $this->verifyDocument($document);

        $this->uow->documentManager->persist($document);
        $this->uow->documentManager->remove($document);
    }

    final protected function scheduleDocumentForUpsert(object $document): void
    {
        $this->verifyDocument($document);

        $this->uow->documentManager->persist($document);
    }

    private function registerDocumentClassNameList(): void
    {
        $className = static::getDocumentClassName();

        try {
            $classMetadata = $this->uow->documentManager->getClassMetadata($className);
        } catch (Throwable $exception) {
            throw MappingException::documentIsNotSet($className, $exception);
        }

        $list = [$className];

        foreach ($classMetadata->associationMappings as $mapping) {
            $own = $mapping['isOwningSide'];
            $ref = $mapping['reference'] ?? false;
            $cls = $mapping['targetDocument'] ?? null;

            if ($ref && $own && $cls !== null) {
                $list[] = $cls;
            }
        }

        $this->documentClassNameList = $list;
    }

    /**
     * @param ?class-string $documentClassName
     *
     * @return class-string
     */
    private function acceptDocumentClassName(?string $documentClassName): string
    {
        if ($documentClassName === null) {
            return $this->documentClassNameList[0];
        }

        if (!in_array($documentClassName, $this->documentClassNameList, true)) {
            throw new LogicException(
                sprintf('Repository %s does not own documents of type %s', static::class, $documentClassName),
            );
        }

        return $documentClassName;
    }

    private function verifyDocument(object $document): void
    {
        // @TODO: Should we allow inserting/removing referenced documents directly or only through the root document?

        if ($document instanceof $this->documentClassNameList[0]) {
            return;
        }

        throw new LogicException(
            sprintf('Document for repository %s must be an instance of %s', static::class, $this->documentClassNameList[0]),
        );
    }
}
