<?php

declare(strict_types=1);

namespace DawidRza\Component\Persistence\MongoDB;

use DawidRza\Component\Persistence\MongoDB\Exception\MappingException;
use Doctrine\ODM\MongoDB\Aggregation\Builder as AggregationBuilder;
use Doctrine\ODM\MongoDB\Query\Builder as QueryBuilder;
use LogicException;
use Throwable;

/**
 * @template TDocument of object
 */
abstract class MongoRepository
{
    /**
     * @var class-string<TDocument>
     */
    private string $documentClassName;

    public function __construct(
        private readonly MongoUOW $uow,
    ) {
        $this->registerDocumentClassName();
    }

    /**
     * @return class-string<TDocument>
     */
    abstract protected static function getDocumentClassName(): string;

    final protected function createAggregationBuilder(): AggregationBuilder
    {
        return $this->uow->documentManager->createAggregationBuilder($this->documentClassName);
    }

    final protected function createQueryBuilder(): QueryBuilder
    {
        return $this->uow->documentManager->createQueryBuilder($this->documentClassName);
    }

    /**
     * @return ?TDocument
     */
    final protected function findDocumentById(mixed $documentId): ?object
    {
        return $this->uow->documentManager->find($this->documentClassName, $documentId);
    }

    final protected function scheduleDocumentForRemove(object $document): void
    {
        $this->verifyDocument($document);

        $this->uow->documentManager->persist($document);
        $this->uow->documentManager->remove($document);
    }

    final protected function scheduleDocumentForUpsert(object $document): void
    {
        $this->verifyDocument($document);

        $this->uow->documentManager->persist($document);
    }

    private function registerDocumentClassName(): void
    {
        $className = static::getDocumentClassName();

        try {
            $this->uow->documentManager->getClassMetadata($className);
        } catch (Throwable $exception) {
            throw MappingException::documentIsNotSet($className, $exception);
        }

        $this->documentClassName = $className;
    }

    private function verifyDocument(object $document): void
    {
        if ($document instanceof $this->documentClassName) {
            return;
        }

        throw new LogicException(
            sprintf('Document for repository %s must be an instance of %s', static::class, $this->documentClassName),
        );
    }
}
