<?php

declare(strict_types=1);

namespace DawidRza\Component\Persistence\MongoDB;

use DawidRza\Component\Persistence\MongoDB\Exception\MappingException;
use Doctrine\ODM\MongoDB\Aggregation\Builder as AggregationBuilder;
use Doctrine\ODM\MongoDB\DocumentManager;
use Doctrine\ODM\MongoDB\Query\Builder as QueryBuilder;
use Doctrine\ODM\MongoDB\Repository\DocumentRepository;
use Doctrine\Persistence\ObjectRepository;
use LogicException;
use Throwable;

/**
 * Simple common parent for all classes that are meant for browsing MongoDB documents.
 */
abstract class MongoReader
{
    /**
     * @note Document manager SHOULD be granted read-only permissions.
     */
    public function __construct(
        private readonly DocumentManager $documentManager,
    ) {
    }

    /**
     * @param class-string $documentName
     */
    final protected function createAggregationBuilder(string $documentName): AggregationBuilder
    {
        $this->verifyDocumentName($documentName);

        return $this->documentManager->createAggregationBuilder($documentName);
    }

    /**
     * @param class-string $documentName
     */
    final protected function createQueryBuilder(string $documentName): QueryBuilder
    {
        $this->verifyDocumentName($documentName);

        return $this->documentManager->createQueryBuilder($documentName);
    }

    /**
     * @template TDocument of object
     *
     * @param class-string<TDocument> $documentName
     *
     * @return ?TDocument
     */
    final protected function findDocumentById(string $documentName, mixed $documentId): ?object
    {
        $this->verifyDocumentName($documentName);

        return $this->documentManager->find($documentName, $documentId);
    }

    /**
     * @param class-string $documentName
     */
    private function verifyDocumentName(string $documentName): void
    {
        /**
         * @var list<class-string> $VERIFIED
         */
        static $VERIFIED = [];

        if (in_array($documentName, $VERIFIED, true)) {
            return;
        }

        try {
            $classMetadata = $this->documentManager->getClassMetadata($documentName);

            if ($classMetadata->isReadOnly !== true) {
                throw new LogicException('Document must be set as read-only.');
            }
        } catch (Throwable $exception) {
            throw new MappingException($documentName, $exception);
        }

        $VERIFIED[] = $documentName;
    }
}
