<?php

declare(strict_types=1);

namespace DawidRza\Bridge\Nexus;

use Closure;
use InvalidArgumentException;
use PhpAmqpLib\Connection\AbstractConnection;
use PhpAmqpLib\Connection\AMQPConnectionConfig;
use PhpAmqpLib\Connection\AMQPConnectionFactory;
use PhpAmqpLib\Message\AMQPMessage;
use PhpAmqpLib\Wire\AMQPTable;
use Throwable;

/**
 * TODO: Add error management (separate method for getting errors: useful for example to log errors and free the error queue)
 * TODO: Add logger
 */
final readonly class Service
{
    /**
     * Key used for indicating a message is going to be redelivered later.
     */
    private const string KEY_AGAIN = '__AGAIN';

    /**
     * Key used for indicating an error occurred and the message must end up in the dead letter queue.
     */
    private const string KEY_ERROR = '__ERROR';

    /**
     * Identifier of the consume channel.
     */
    private const int CHANNEL_CONSUME = 32;

    /**
     * Identifier of the publish channel.
     */
    private const int CHANNEL_PUBLISH = 64;

    private AbstractConnection $connection;

    private string $name;

    public function __construct(
        string $host,
        int $port,
        string $name,
        string $secret,
    ) {
        $configuration = new AMQPConnectionConfig();
        $configuration->setHost($host);
        $configuration->setPort($port);
        $configuration->setUser($name);
        $configuration->setPassword($secret);
        $configuration->setIsLazy(true);

        $this->connection = AMQPConnectionFactory::create($configuration);
        $this->name = $name;
    }

    /**
     * @param callable(Envelope): void $callback
     */
    public function consume(callable $callback): void
    {
        $channel = $this->connection->channel(self::CHANNEL_CONSUME);

        $channel->basic_qos(
            prefetch_size: 0,
            prefetch_count: 5,
            a_global: false
        );

        $channel->basic_consume(
            queue: $this->name,
            callback: $this->getClosureForConsume($callback),
        );

        // Default maximum poll interval is 10 seconds.
        $channel->consume();
    }

    /**
     * @param array<string, bool|float|int|null|string> $headers
     */
    public function publish(string $type, array $payload, array $headers = [], int $retries = 5): void
    {
        if ($type === "") {
            throw new InvalidArgumentException('Message type is empty');
        }

        $body = json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

        if ($body === false) {
            throw new InvalidArgumentException('Message payload could not be encoded as valid JSON.');
        }

        try {
            $headers = new AMQPTable(array_merge($headers, [
                'nx_retry' => $retries, // TODO: Maybe we can allow to set this per service as a configurable default?
            ]));
        } catch (Throwable $exception) {
            throw new InvalidArgumentException('Message headers coult not be encoded', previous: $exception);
        }

        $message = new AMQPMessage(
            body: $body,
            properties: [
                'app_id' => $this->name,
                'application_headers' => $headers,
                'content_type' => 'application/json',
                'content_encoding' => 'UTF-8',
                'delivery_mode' => AMQPMessage::DELIVERY_MODE_PERSISTENT,
                'timestamp' => (int) (microtime(true) * 1000), // milliseconds
                'type' => $type,
                'message_id' => bin2hex(random_bytes(16)),
            ],
        );

        $this->connection->channel(self::CHANNEL_PUBLISH)->basic_publish(
            msg: $message,
            exchange: $this->name,
            routing_key: $type,
        );
    }

    public function stopConsume(): void
    {
        $this->connection->channel(self::CHANNEL_CONSUME)->stopConsume();
    }

    private function again(AMQPMessage $message, Throwable $throwable): void
    {
        $headers = $this->getHeaders($message);

        $retry = (int) $headers['nx_retry'];

        if ($retry <= 0) {
            $this->error($message, errorType: Error::SERVICE_ERROR, errorText: $throwable->getMessage());
            return;
        }

        $headers['nx_retry'] = $retry - 1;

        $this->connection->channel(self::CHANNEL_PUBLISH)->basic_publish(
            msg: $message,
            exchange: $this->name,
            routing_key: self::KEY_AGAIN,
        );

        $message->nack();
    }

    private function error(AMQPMessage $message, Error $errorType, ?string $errorText = null): void
    {
        $headers = $this->getHeaders($message);

        $headers['nx_error_type'] = $errorType->name;
        $headers['nx_error_text'] = substr($errorText ?? "", 0, 512);

        $this->connection->channel(self::CHANNEL_PUBLISH)->basic_publish(
            msg: $message,
            exchange: $this->name,
            routing_key: self::KEY_ERROR,
        );

        $message->nack();
    }

    /**
     * @param callable(Envelope): void $fn
     *
     * @return Closure(AMQPMessage): void
     */
    private function getClosureForConsume(callable $fn): Closure
    {
        return function (AMQPMessage $message) use ($fn) {
            $payload = json_decode($message->getBody(), associative: true);

            if ($payload === null) {
                $this->error($message, errorType: Error::INVALID_PAYLOAD_JSON, errorText: json_last_error_msg());
                return;
            }

            if (!$message->has('app_id')) {
                $this->error($message, errorType: Error::INVALID_MESSAGE_NO_SERVICE);
                return;
            }

            if (!$message->has('type')) {
                $this->error($message, errorType: Error::INVALID_MESSAGE_NO_TYPE);
                return;
            }

            if (!$message->has('message_id')) {
                $this->error($message, errorType: Error::INVALID_MESSAGE_NO_ID);
                return;
            }

            if (!$message->has('timestamp')) {
                $this->error($message, errorType: Error::INVALID_MESSAGE_NO_TIME);
                return;
            }

            if (!$message->has('application_headers')) {
                $headers = new AMQPTable([]);
            } else {
                $headers = $message->get('application_headers');

                unset(
                    $headers['x-death'],
                    $headers['x-first-death-exchange'],
                    $headers['x-first-death-queue'],
                    $headers['x-first-death-reason'],
                    $headers['x-last-death-exchange'],
                    $headers['x-last-death-queue'],
                    $headers['x-last-death-reason'],
                );
            }

            $envelope = new Envelope(
                id: $message->get('message_id'),
                service: $message->get('app_id'),
                type: $message->get('type'),
                time: $message->get('timestamp'),
                payload: $payload,
                headers: $headers->getNativeData(),
            );

            try {
                $fn($envelope);
            } catch (Throwable $exception) {
                $this->again($message, throwable: $exception);
                return;
            }

            $message->ack();
        };
    }

    private function getHeaders(AMQPMessage $message): AMQPTable
    {
        if ($message->has('application_headers')) {
            return $message->get('application_headers');
        }

        $headers = new AMQPTable([]);

        $message->set('application_headers', $headers);

        return $headers;
    }
}
