<?php

declare(strict_types=1);

namespace DawidRza\Component\Http\Symfony;

use DawidRza\Component\Http\ContentType;
use DawidRza\Component\Http\Context;
use DawidRza\Component\Http\ContextBinder;
use DawidRza\Component\Http\ContextBinderTrait;
use DawidRza\Component\Http\Response;
use SplFileInfo;
use Symfony\Component\HttpFoundation\Request;
use Undefined;

final class SymfonyContext implements Context
{
    use ContextBinderTrait;

    /**
     */
    private ?array $jsonContent = null;

    /**
     */
    private ?Response $response = null;

    public function __construct(
        private readonly ContextBinder $contextBinder,
        private readonly Request $request,
    ) {
    }

    /**
     * {@inheritDoc}
     */
    public function body(): ?string
    {
        return $this->request->getContent() ?: null;
    }

    /**
     * {@inheritDoc}
     */
    public function contentType(): ?ContentType
    {
        $contentType = $this->request->getContentTypeFormat();

        return null === $contentType
             ? null
             : ContentType::tryFromString($this->request->getContentTypeFormat());
    }

    /**
     * {@inheritDoc}
     */
    public function formValue(string $name, ?string $default = null): ?string
    {
        $value = $this->request->request->get($name);

        return is_string($value)
             ? $value
             : $default;
    }

    /**
     * {@inheritDoc}
     */
    public function file(string $name): ?SplFileInfo
    {
        $value = $this->request->files->get($name);

        return $value instanceof SplFileInfo
             ? $value
             : null;
    }

    /**
     * {@inheritDoc}
     */
    public function header(string $name, ?string $default = null): ?string
    {
        return $this->request->headers->get($name) ?: $default;
    }

    /**
     * {@inheritDoc}
     */
    public function jsonValue(string $name): Undefined|array|bool|float|int|string|null
    {
        $this->parseJSON();

        return array_key_exists($name, $this->jsonContent)
             ? $this->jsonContent[$name]
             : UNDEFINED;
    }

    /**
     * {@inheritDoc}
     */
    public function pathValue(string $name, ?string $default = null): ?string
    {
        $value = $this->request->attributes->get($name);

        return is_string($value)
             ? $value
             : $default;
    }

    /**
     * {@inheritDoc}
     */
    public function queryValue(string $name, ?string $default = null): ?string
    {
        $value = $this->request->query->get($name);

        return is_string($value)
             ? $value
             : $default;
    }

    /**
     * {@inheritDoc}
     */
    public function realIP(): ?string
    {
        return $this->request->getClientIp();
    }

    /**
     * {@inheritDoc}
     */
    public function response(): Response
    {
        return $this->response ??= new Response();
    }

    /**
     * @see ContextBinderTrait::getBinder()
     */
    private function getBinder(): ContextBinder
    {
        return $this->contextBinder;
    }

    /**
     */
    private function parseJSON(): void
    {
        if ($this->jsonContent !== null) {
            return;
        }

        if ($this->contentType() !== ContentType::JSON) {
            throw new \Exception(); // todo
        }

        $this->jsonContent = $this->request->toArray(); // todo handle error
    }
}
