<?php

declare(strict_types=1);

namespace DawidRza\Component\Http;

use SplFileInfo;
use Undefined;

interface Context
{
    /**
     * Attempts to create an instance of a $class and bind request's data to that instance.
     *
     * @template TReturn
     *
     * @param class-string<TReturn> $class
     *
     * @return TReturn
     *
     *  @throws TODO
     */
    public function bind(string $class): object;

    /**
     * Returns raw body from the request.
     *
     * This method MUST return NULL when body is empty.
     */
    public function body(): ?string;

    /**
     * Returns this request's content type based on the Content-Type header.
     *
     * This method MUST return NULL if content type is either missing or not supported.
     */
    public function contentType(): ?ContentType;

    /**
     * Returns file from the request.
     *
     * This method MUST return null if file does not exist within the request.
     */
    public function file(string $name): ?SplFileInfo;

    /**
     * Returns form parameter value from the request.
     *
     * If default is not set, this method MUST return NULL when form parameter is either missing or empty.
     */
    public function formValue(string $name, ?string $default = null): ?string;

    /**
     * Returns header from the request.
     *
     * If default is not set, this method MUST return NULL when header is either missing or empty.
     */
    public function header(string $name, ?string $default = null): ?string;

    /**
     * Returns a single value from the request's JSON payload.
     *
     * This method MUST return UNDEFINED if content type is other than ContentType::JSON.
     * This method MUST return UNDEFINED if key is missing.
     */
    public function jsonValue(string $name): Undefined|array|bool|float|int|null|string;

    /**
     * Returns path parameter value from the request.
     *
     * If default is not set, this method MUST return NULL when path parameter is either missing or empty.
     */
    public function pathValue(string $name, ?string $default = null): ?string;

    /**
     * Returns query parameter value from the request.
     *
     * If default is not set, this method MUST return NULL when query parameter is either missing or empty.
     */
    public function queryValue(string $name, ?string $default = null): ?string;

    /**
     * Returns a real remote IP address.
     *
     * This method MUST return NULL if remote IP is either missing or request is not coming from a trusted proxy.
     */
    public function realIP(): ?string;

    /**
     * Returns a Response object associated with this request.
     */
    public function response(): Response;
}
