<?php

declare(strict_types=1);

namespace DawidRza\Component\Http\Symfony\DependencyInjection;

use DawidRza\Component\Http\ContextBinder;
use DawidRza\Component\Http\Json\HydratorRegistry;
use DawidRza\Component\Http\Json\JsonMapper;
use DawidRza\Component\Http\Symfony\EventListener\OnRequest;
use DawidRza\Component\Http\Symfony\EventListener\OnView;
use DawidRza\Component\Http\Symfony\SymfonyValidator;
use DawidRza\Component\Http\Symfony\ValueResolver\ContextValueResolver;
use Symfony\Component\DependencyInjection\Compiler\CompilerPassInterface;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Definition;
use Symfony\Component\DependencyInjection\Extension\AbstractExtension;
use Symfony\Component\DependencyInjection\Loader\Configurator\ContainerConfigurator;
use Symfony\Component\DependencyInjection\Reference;
use Symfony\Component\Validator\Validator\ValidatorInterface;

final class SymfonyContextExtension extends AbstractExtension implements CompilerPassInterface
{
    public function __construct(
        private readonly bool $useResponse = true,
        private readonly bool $useResolver = true,
    ) {
    }

    /**
     * {@inheritDoc}
     */
    public function loadExtension(array $config, ContainerConfigurator $container, ContainerBuilder $builder): void
    {
        $services = $container->services();

        $services
            ->defaults()
            ->autoconfigure()
            ->autowire();

        $this->registerContextBinder($container);
        $this->registerJsonMapper($container);

        $services->set(OnRequest::class);

        if ($this->useResponse) {
            $services->set(OnView::class);
        }

        if ($this->useResolver) {
            $services->set(ContextValueResolver::class);
        }
    }

    public function process(ContainerBuilder $container): void
    {
        if (!$container->hasAlias(ValidatorInterface::class)) {
            return;
        }

        $container->setDefinition(
            id: SymfonyValidator::class,
            definition: new Definition(SymfonyValidator::class, [
                '$validator' => new Reference(ValidatorInterface::class),
            ]),
        );

        $container
            ->getDefinition(ContextBinder::class)
            ->setArgument('$validator', new Reference(SymfonyValidator::class));
    }

    private function registerContextBinder(ContainerConfigurator $container): void
    {
        $container
            ->services()
            ->set(ContextBinder::class)
            ->arg('$classMetadata', null)
            ->arg('$validator', null);
    }

    private function registerJsonMapper(ContainerConfigurator $container): void
    {
        $services = $container->services();

        $services
            ->set(HydratorRegistry::class)
            ->arg('$cacheDir', '%kernel.cache_dir%/dawidrza/json');

        $services
            ->set(JsonMapper::class)
            ->arg('$hydrators', new Reference(HydratorRegistry::class));
    }
}
