<?php

declare(strict_types=1);

namespace DawidRza\Component\Http\Metadata;

use InvalidArgumentException;
use ReflectionIntersectionType;
use ReflectionNamedType;
use ReflectionProperty;
use Undefined;

final readonly class PropertyType
{
    public function __construct(
        public string $name,
        public bool $isRequired,
        public bool $isNullable,
        public bool $isEnum,
    ) {
    }

    public static function parse(ReflectionProperty $property): self
    {
        $reflectionType = $property->getType();

        if (null === $reflectionType) {
            throw new InvalidArgumentException('Bound property must be type-hinted.');
        }

        if ($reflectionType instanceof ReflectionIntersectionType) {
            throw new InvalidArgumentException('Bound property must not be type-hinted with intersection type.');
        }

        if ($reflectionType instanceof ReflectionNamedType) {
            $name = $reflectionType->getName();

            return new self(
                name: $name,
                isRequired: $name !== Undefined::class,
                isNullable: $reflectionType->allowsNull(),
                isEnum: enum_exists($name),
            );
        }

        $name = null;
        $isRequired = true;

        foreach ($reflectionType->getTypes() as $type) {
            if (!$type instanceof ReflectionNamedType) {
                throw new InvalidArgumentException('Bound property type-hinted with union type must consist only of named types.');
            }

            if ($type->getName() === 'null') {
                continue;
            }

            if ($type->getName() === Undefined::class) {
                $isRequired = false;
                continue;
            }

            if ($name !== null) {
                throw new InvalidArgumentException('Bound property type-hinted with union type must consist only of one named type other than \Undefined and null');
            }

            $name = $type->getName();
        }

        return new self(
            name: $name,
            isRequired: $isRequired,
            isNullable: $reflectionType->allowsNull(),
            isEnum: enum_exists($name),
        );
    }
}
