<?php

declare(strict_types=1);

namespace DawidRza\Component\Http\Json\Metadata;

use DawidRza\Component\Http\Exception\DefinitionException;
use ReflectionClass;
use ReflectionEnum;
use ReflectionIntersectionType;
use ReflectionNamedType;
use ReflectionProperty;
use ReflectionType;
use ReflectionUnionType;
use Undefined;

final class Property
{
    public function __construct(
        public string $name,
        public ?string $typeName,
        public ?string $enumType,
        public bool $hasDefault,
        public bool $isNullable,
        public bool $isObject,
        public bool $isPromoted,
        public bool $isRequired,
    ) {
    }

    public static function fromReflection(ReflectionProperty $reflection): self
    {
        $reflectionType = $reflection->getType();

        if ($reflectionType === null) {
            throw self::error($reflection, "Missing property type");
        }

        $property = new self(
            name: $reflection->getName(),
            typeName: null,
            enumType: null,
            hasDefault: false,
            isNullable: false,
            isObject: false,
            isPromoted: false,
            isRequired: true,
        );

        self::resolveType($property, $reflection, $reflectionType);

        if ($reflection->hasDefaultValue()) {
            $property->hasDefault = true;
            $property->isRequired = false;
        }

        if ($reflection->isPromoted()) {
            $property->isPromoted = true;
        }

        if ($property->typeName === null) {
            return $property;
        }

        // Enum check must come first because class_exists() returns true with enums too.
        if (enum_exists($property->typeName)) {
            $reflectionEnum = new ReflectionEnum($property->typeName);

            if (!$reflectionEnum->isBacked()) {
                throw self::error($reflection, "Enum property must be a backed enum");
            }

            $property->enumType = $property->typeName;
            $property->typeName = $reflectionEnum->getBackingType()->getName();

            return $property;
        }

        if (class_exists($property->typeName)) {
            $reflectionClass = new ReflectionClass($property->typeName);

            if (!$reflectionClass->isUserDefined()) {
                throw self::error($reflection, "Object property must be a user-defined object");
            }

            $property->isObject = true;

            return $property;
        }

        return $property;
    }

    private static function error(ReflectionProperty $reflection, string $message): DefinitionException
    {
        return new DefinitionException(
            sprintf('[JSON] %s::$%s - %s', $reflection->getDeclaringClass()->getName(), $reflection->getName(), $message),
        );
    }

    private static function resolveType(
        self $property,
        ReflectionProperty $reflectionProperty,
        ReflectionType $reflectionType,
    ): void {
        if ($reflectionType instanceof ReflectionIntersectionType) {
            throw self::error($reflectionProperty, "Property intersection type is not supported");
        }

        if ($reflectionType instanceof ReflectionUnionType) {
            $subTypes = $reflectionType->getTypes();
        } else {
            $subTypes = [$reflectionType];
        }

        foreach ($subTypes as $subType) {
            if (!$subType instanceof ReflectionNamedType) {
                throw self::error($reflectionProperty, 'Property type must consist only of one named type other than Undefined and null');
            }

            $typeName = $subType->getName();

            switch ($typeName) {
                // When type is mixed, we might as well allow anything and just stop here.
                case 'mixed':
                    $property->isNullable = true;
                    $property->isRequired = false;
                    $property->typeName = null;
                    return;

                case Undefined::class:
                    $property->isRequired = false;
                    continue 2;

                case 'null':
                    $property->isNullable = true;
                    continue 2;

                default:
                    break;
            }

            if ($property->typeName !== null) {
                throw self::error($reflectionProperty, 'Property type must consist only of one named type other than Undefined and null');
            }

            $property->isNullable = $reflectionType->allowsNull();
            $property->typeName = $typeName;
        }

        if ($property->typeName === null) {
            throw self::error($reflectionProperty, 'Property is missing type other than Undefined and null');
        }
    }
}
