<?php

declare(strict_types=1);

namespace DawidRza\Component\Http\Metadata;

use DawidRza\Component\Http\Metadata\Attribute\Bind;
use InvalidArgumentException;
use ReflectionClass;

final readonly class ClassMetadata
{
    private const array VARS = [
        'header' => null,
        'path' => null,
        'query' => null,
        'file' => null,
    ];

    /**
     * @param list<PropertyMetadata> $properties
     */
    public function __construct(
        public string $name,
        public array $properties,
    ) {
    }

    public static function parse(string $class): self
    {
        if (!class_exists($class)) {
            throw new InvalidArgumentException("Class does not exist: $class");
        }

        $reflection = new ReflectionClass($class);

        $props = [];

        foreach ($reflection->getProperties() as $property) {
            $body = [];
            $vars = self::VARS;

            $attributes = $property->getAttributes(Bind::class);

            if (count($attributes) === 0) {
                throw new InvalidArgumentException(
                    sprintf("Property is not bound to the request: %s::$%s", $class, $property->getName())
                );
            }

            foreach ($attributes as $attribute) {
                /** @var Bind $bind */
                $bind = $attribute->newInstance();

                if (array_key_exists($bind->from, $vars)) {
                    $vars[$bind->from] = $bind;
                } else {
                    $body[$bind->from] = $bind;
                }
            }

            $props[] = new PropertyMetadata(
                name: $property->getName(),
                type: PropertyType::parse($property),
                vars: array_filter($vars),
                body: $body,
            );
        }

        return new self(
            $reflection->getName(),
            $props,
        );
    }
}
