<?php

declare(strict_types=1);

namespace DawidRza\Component\Http\Json;

use DawidRza\Component\Http\Exception\DefinitionException;
use DawidRza\Component\Http\Json\Metadata\Property;
use Nette\PhpGenerator\ClassType;
use Nette\PhpGenerator\Method;
use Nette\PhpGenerator\PhpFile;
use Nette\PhpGenerator\Printer;
use ReflectionClass;

/**
 * @internal
 */
final readonly class HydratorGenerator
{
    public function __construct(
        private HydratorRegistry $registry,
        private string $cacheDir,
    ) {
    }

    /**
     * @template T of object
     *
     * @param class-string<T> $class
     *
     * @return Hydrator<T>
     */
    public function create(string $class): Hydrator
    {
        if (!class_exists($class)) {
            throw new DefinitionException(
                sprintf('[JSON] %s - Class does not exist', $class),
            );
        }

        $className = $this->generateHydratorClass(
            new ReflectionClass($class),
        );

        return new $className($this->registry);
    }

    /**
     * @template T of object
     *
     * @param ReflectionClass<T> $reflectionClass
     *
     * @return class-string<T>
     */
    private function generateHydratorClass(ReflectionClass $reflectionClass): string
    {
        $className = sprintf("Hydrator_%s", md5($reflectionClass->getName()));
        $cachePath = "$this->cacheDir/$className.php";

        // Hydrator has been already loaded.
        if (class_exists($className) && is_a($className, Hydrator::class, true)) {
            return $className;
        }

        //> FILE & CLASS

        $netteFile = new PhpFile();
        $netteFile
            ->setStrictTypes()
            ->addUse(Error::class);

        $netteClass = $netteFile->addClass($className);
        $netteClass
            ->setExtends(Hydrator::class);

        $netteFunction = $netteClass
            ->addMethod('doHydrate')
            ->setProtected();
        $netteFunction
            ->addParameter('source')
            ->setType('array');
        $netteFunction
            ->addParameter('errors')
            ->setReference()
            ->setType('array');
        $netteFunction
            ->setReturnType($reflectionClass->getName());

        //> STATIC INITIALIZATION

        $netteFunction
            ->addBody('static $reflection = new ReflectionClass(?);', [$reflectionClass->getName()]);

        $netteFunction
            ->addBody('')
            ->addBody('$object = $reflection->newInstanceWithoutConstructor();')
            ->addBody('$errors = [];')
            ->addBody('');

        //> PROPERTIES

        foreach ($reflectionClass->getProperties() as $reflectionProperty) {
            $property = Property::fromReflection($reflectionProperty);

            $this->generatePropertyHydrator($netteClass, $property);

            $netteFunction
                ->addBody('$e = $this->?($reflection, $object, $source);', [$property->name])
                ->addBody('if ($e !== null) {')
                ->addBody("\t\$errors[?] = is_array(\$e) \? \$e : \$e->name;", [$property->name])
                ->addBody('}')
                ->addBody('');
        }

        //> RETURN

        $netteFunction
            ->addBody('return $object;');

        //> PRINT

        $printer = new Printer();
        $printer->indentation = '    ';

        $code = $printer->printFile($netteFile);

        if (!is_dir($this->cacheDir)) {
            mkdir($this->cacheDir, recursive: true);
        }

        file_put_contents($cachePath, $code);

        require_once $cachePath;

        return $className;
    }

    private function generatePropertyHydrator(ClassType $netteClass, Property $property): void
    {
        $netteFunction = $this->generatePropertyHydratorMethod($netteClass, $property);

        // > UNDEFINED

        if ($property->isRequired) {
            $netteFunction
                ->addBody('if (!array_key_exists(?, $source)) {', [$property->name])
                ->addBody("\treturn Error::REQUIRED;")
                ->addBody('}');
        } else {
            $netteFunction
                ->addBody('if (!array_key_exists(?, $source)) {', [$property->name]);

            if (!$property->hasDefault) {
                $netteFunction
                    ->addBody("\t\$reflector->getProperty(?)->setValue(\$target, UNDEFINED);", [$property->name]);
            }

            $netteFunction
                ->addBody("\treturn null;")
                ->addBody('}');
        }

        //> NULL & TYPE-CHECK
        $netteFunction
            ->addBody('')
            ->addBody('$value = $source[?];', [$property->name])
            ->addBody('');

        if ($property->typeName !== null) {
            if ($property->isNullable) {
                $netteFunction
                    ->addBody('if ($value === null) {')
                    ->addBody("\t\$reflector->getProperty(?)->setValue(\$target, null);", [$property->name])
                    ->addBody("\treturn null;")
                    ->addBody('}');
            }

            $this->generatePropertyTypeCheck($netteFunction, $property);
        }

        $netteFunction
            ->addBody('')
            ->addBody('$reflector->getProperty(?)->setValue($target, $value);', [$property->name])
            ->addBody('return null;');
    }

    private function generatePropertyHydratorMethod(ClassType $netteClass, Property $property): Method
    {
        $netteFunction = $netteClass->addMethod($property->name);
        $netteFunction
            ->setReturnType('Error|array')
            ->setReturnNullable()
            ->setPrivate();
        $netteFunction
            ->addParameter('reflector')->setType(ReflectionClass::class);
        $netteFunction
            ->addParameter('target')->setType('object');
        $netteFunction
            ->addParameter('source')->setType('array');

        return $netteFunction;
    }

    private function generatePropertyTypeCheck(Method $netteFunction, Property $property): void
    {
        switch ($property->typeName) {
            case 'string':
                $netteFunction
                    ->addBody("if (!is_string(\$value)) {")
                    ->addBody("\treturn Error::EXPECT_STRING;")
                    ->addBody("}");
                break;
            case 'int':
                $netteFunction
                    ->addBody("if (!is_int(\$value)) {")
                    ->addBody("\treturn Error::EXPECT_INT;")
                    ->addBody("}");
                break;
            case 'float':
                $netteFunction
                    ->addBody("if (!is_float(\$value)) {")
                    ->addBody("\treturn Error::EXPECT_FLOAT;")
                    ->addBody("}");
                break;
            case 'true':
                $netteFunction
                    ->addBody("if (\$value !== true) {")
                    ->addBody("\treturn Error::EXPECT_TRUE;")
                    ->addBody("}");
                break;
            case 'false':
                $netteFunction
                    ->addBody("if (\$value !== false) {")
                    ->addBody("\treturn Error::EXPECT_FALSE;")
                    ->addBody("}");
                break;
            case 'bool':
                $netteFunction
                    ->addBody("if (!is_bool(\$value)) {")
                    ->addBody("\treturn Error::EXPECT_BOOL;")
                    ->addBody("}");
                break;
            case 'array':
                $netteFunction
                    ->addBody("if (!is_array(\$value)) {")
                    ->addBody("\treturn Error::EXPECT_ARRAY;")
                    ->addBody("}");
                break;
        }

        if ($property->enumType !== null) {
            $netteFunction
                ->addBody('')
                ->addBody('$value = call_user_func([?, "tryFrom"], $value);', [$property->enumType])
                ->addBody('if ($value === null) {')
                ->addBody("\treturn Error::INVALID_ENUM;")
                ->addBody('}');
            return;
        }

        if ($property->isObject) {
            // Warm-up
            $this->registry->get($property->typeName);

            $netteFunction
                ->addBody("if (!is_array(\$value)) {")
                ->addBody("\treturn Error::EXPECT_OBJECT;")
                ->addBody('}')
                ->addBody('')
                ->addBody('$errors = [];')
                ->addBody('$value = $this->registry->get(?)->doHydrate($value, $errors);', [$property->typeName])
                ->addBody('if (count($errors) > 0) {')
                ->addBody("\treturn \$errors;")
                ->addBody('}');
        }
    }
}
