<?php

declare(strict_types=1);

namespace DawidRza\Component\Http;

use DawidRza\Component\Http\Exception\BindException;
use DawidRza\Component\Http\Exception\BindError;
use DawidRza\Component\Http\Exception\UnexpectedErrorException;
use DawidRza\Component\Http\Metadata\Attribute\Bind;
use DawidRza\Component\Http\Metadata\ClassMetadataRegistry;
use DawidRza\Component\Http\Validator\NullValidator;
use DawidRza\Component\Http\Validator\Validator;
use ReflectionClass;
use ReflectionException;
use Throwable;
use TypeError;

final readonly class ContextBinder
{
    private ClassMetadataRegistry $classMetadataRegistry;

    private Validator $validator;

    public function __construct(?ClassMetadataRegistry $classMetadata, ?Validator $validator)
    {
        $this->classMetadataRegistry = $classMetadata ?? new ClassMetadataRegistry();
        $this->validator = $validator ?? new NullValidator();
    }

    /**
     * @template TReturn
     *
     * @param class-string<TReturn> $class
     *
     * @return TReturn
     *
     * @throws BindException
     * @throws BindInputException
     */
    public function bind(Context $context, string $class): object
    {
        $classMetadata = $this->classMetadataRegistry->get($class);

        try {
            $reflection = new ReflectionClass($class);
        } catch (ReflectionException $exception) {
            throw new UnexpectedErrorException($exception);
        }

        // TODO: This should be allowed but without private properties in parent class
        // todo class metadata check if internal
//        if ($reflection->getParentClass() !== false) {
//            throw new BindException(
//                error: Error::ERROR_PARENT_CLASS_UNSUPPORTED,
//                class: $classMetadata,
//            );
//        }


        try {
            // todo class metadata check if internal
            $instance = $reflection->newInstanceWithoutConstructor();
        } catch (ReflectionException $exception) {
            throw new UnexpectedErrorException($exception);
        }

        $bodyType = $context->contentType();

        foreach ($classMetadata->properties as $propertyMetadata) {
            try {
                // todo class metadata check if internal
                $propertyReflection = $reflection->getProperty($propertyMetadata->name);
            } catch (ReflectionException $exception) {
                throw new UnexpectedErrorException($exception);
            }

            $propertyType = $propertyMetadata->type;

            $binds = $propertyMetadata->bindsByContentType($bodyType);

            if (empty($binds)) {
                BindError::UNSUPPORTED_CONTENT_TYPE->raise($classMetadata, $propertyMetadata);
            }

            $valueFromContext = UNDEFINED;

            foreach ($binds as $bind) {
                $value = $this->getValue($context, $bind);

                if ($value === UNDEFINED) {
                    continue;
                }

                if ($value !== null && $propertyType->isEnum) {
                    try {
                        $value = call_user_func_array([$propertyType->name, 'from'], [$value]);
                    } catch (Throwable) {
                        BindError::INVALID_ENUM->raise($classMetadata, $propertyMetadata);
                    }
                }

                $valueFromContext = $value;
                break;
            }

            if (is_undefined($valueFromContext) && $propertyType->isRequired) {
                BindError::VALUE_REQUIRED->raise($classMetadata, $propertyMetadata);
            }

            try {
                $propertyReflection->setValue($instance, $valueFromContext);
            } catch (TypeError $exception) {
                BindError::INVALID_TYPE->raise($classMetadata, $propertyMetadata, $exception);
            } catch (Throwable $exception) {
                BindError::UNKNOWN->raise($classMetadata, $propertyMetadata, $exception);
            }
        }

        $this->validator->validate($instance);

        return $instance;
    }

    private function getValue(Context $context, Bind $bind): mixed
    {
        $name = $bind->name;

        return match ($bind->from) {
            Bind::FORM => $context->formValue($name),
            Bind::FILE => $context->file($name),
            Bind::HEADER => $context->header($name),
            Bind::JSON => $context->jsonValue($name),
            Bind::PATH => $context->pathValue($name),
            Bind::QUERY => $context->queryValue($name),
        };
    }
}
