<?php

declare(strict_types=1);

namespace DawidRza\Component\EmailClient\Type;

/**
 * @property-read array<int, Attachment> $attachments
 * @property-read array<string, Content> $body
 * @property-read array<string, string|list<string>> $headers
 * @property-read array<int, Address> $recipients
 * @property-read ?Address $sender
 * @property-read ?string $subject
 */
final class Message
{
    use ReadonlyTrait;

    /** @var array<int, Attachment> */
    private $attachments;

    /** @var array<string, Content> */
    private $body;

    /** @var array<string, string|list<string>> */
    private $headers;

    /** @var array<int, Address> */
    private $recipients;

    /** @var ?Address */
    private $sender;

    /** @var ?string */
    private $subject;

    private function __construct()
    {
        $this->attachments = [];
        $this->body = [];
        $this->headers = [];
        $this->recipients = [];
        $this->sender = null;
        $this->subject = null;
    }

    public static function empty(): self
    {
        return new self();
    }

    public function withAttachment(Attachment $attachment): self
    {
        $this->attachments[] = $attachment;

        return $this;
    }

    /**
     * @param list<Attachment> $attachments
     */
    public function withAttachments(array $attachments): self
    {
        foreach ($attachments as $attachment) {
            $this->withAttachment($attachment);
        }

        return $this;
    }

    /**
     * @param list<Content> $body
     */
    public function withBody(array $body): self
    {
        foreach ($body as $content) {
            $this->withContent($content);
        }

        return $this;
    }

    public function withContent(Content $content): self
    {
        $this->body[$content->type] = $content;

        return $this;
    }

    public function withRecipient(Address $recipient): self
    {
        $this->recipients[] = $recipient;

        return $this;
    }

    /**
     * @param list<Address> $recipients
     */
    public function withRecipients(array $recipients): self
    {
        foreach ($recipients as $recipient) {
            $this->withRecipient($recipient);
        }

        return $this;
    }

    public function withMessageId(string $messageId): self
    {
        $this->headers['messageId'] = $messageId;

        return $this;
    }

    public function withInReplyTo(string $inReplyTo): self
    {
        $this->headers['inReplyTo'] = $inReplyTo;

        return $this;
    }

    /**
     * @param list<string> $references
     */
    public function withReferences(array $references): self
    {
        $this->headers['references'] = $references;

        return $this;
    }

    public function withSender(Address $sender): self
    {
        $this->sender = $sender;

        return $this;
    }

    public function withSubject(string $subject): self
    {
        $this->subject = $subject;

        return $this;
    }
}
