<?php

declare(strict_types=1);

namespace DawidRza\Component\EmailClient\Symfony;

use DawidRza\Component\EmailClient\Exception\ConnectionUnestablishedException;
use DawidRza\Component\EmailClient\Exception\TransportException;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\ConnectException;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Exception\TransferException;
use LogicException;

/**
 * @phpstan-type SendRequestOutput array{
 *     statusCode: int,
 *     responseBody: array<mixed>,
 * }
 */
final class GuzzleHttpClient
{
    /** @var Client */
    private $guzzleClient;

    public function __construct(
        string $baseUri,
        string $clientId,
        string $clientSecret
    ) {
        $this->guzzleClient = new Client([
            'base_uri' => $baseUri,
            'headers' => [
                'Authorization' => 'Basic ' . base64_encode($clientId . ':' . $clientSecret),
            ],
            'http_errors' => false,
        ]);
    }

    /**
     * @param array<string, mixed> $json
     * @param array<string, null|int|string> $query
     * @return SendRequestOutput
     *
     * @throws ConnectionUnestablishedException When the connection to the mailer could not be established.
     * @throws TransportException When the transport of the request was unsuccessful.
     */
    public function sendRequest(string $method, string $uri, array $query = [], array $json = []): array
    {
        $path = $uri;
        $query = array_filter($query, static function ($value) { return $value !== null; });
        if (count($query) !== 0) {
            $path .= '?' . http_build_query($query);
        }

        try {
            $response = $this->guzzleClient->request(
                $method,
                $path,
                [
                    'json' => $json,
                ]
            );
        } catch (GuzzleException $exception) {
            switch (get_class($exception)) {
                case ConnectException::class:
                case TransferException::class:
                    throw new ConnectionUnestablishedException($exception->getMessage());
                default:
                    throw new LogicException(sprintf(
                        'The email client threw an exception of class %s, which was not caught.',
                        get_class($exception),
                    ));
            }
        }

        $responseBody = [];
        if (in_array('application/json', $response->getHeader('Content-Type'), true)) {
            $responseBody = json_decode($response->getBody()->getContents(), true);
            if (!is_array($responseBody)) {
                throw TransportException::unexpectedResponseFormat(
                    'Expected the response body to be decodable to JSON.'
                );
            }
        }

        return [
            'statusCode' => $response->getStatusCode(),
            'responseBody' => $responseBody,
        ];
    }
}
