<?php

declare(strict_types=1);

namespace DawidRza\Component\EmailClient\Symfony;

use DawidRza\Component\EmailClient\EmailAccount;
use DawidRza\Component\EmailClient\EmailClient;
use DawidRza\Component\EmailClient\Exception\TransportException;
use DawidRza\Component\EmailClient\Service\EmailClientTransformer;
use DawidRza\Component\EmailClient\Type\Account\IncomingServer;
use DawidRza\Component\EmailClient\Type\Account\OutgoingServer;
use DawidRza\Component\EmailClient\Type\Discovery\DiscoveryInformation;

final class GuzzleEmailClient implements EmailClient
{
    /** @var GuzzleHttpClient */
    private $httpClient;

    public function __construct(
        GuzzleHttpClient $httpClient
    ) {
        $this->httpClient = $httpClient;
    }

    public function discoverConfiguration(string $domain): ?DiscoveryInformation
    {
        $response = $this->httpClient->sendRequest(
            'GET',
            "/discovery/$domain",
        );

        switch ($response['statusCode']) {
            case 200:
                return EmailClientTransformer::decomposeBodyFromDiscoverConfiguration($response['responseBody']);
            case 404:
                return null;
            default:
                throw TransportException::unexpectedResponseCode('Expected a 200 or 404 response code.');
        }
    }

    public function registerAccount(IncomingServer $incomingServer, OutgoingServer $outgoingServer): string
    {
        $response = $this->httpClient->sendRequest(
            'POST',
            '/accounts',
            [],
            EmailClientTransformer::composeBodyForRegisterAccount($incomingServer, $outgoingServer),
        );

        switch ($response['statusCode']) {
            case 201:
                return EmailClientTransformer::decomposeBodyFromRegisterAccount($response['responseBody']);
            default:
                throw TransportException::unexpectedResponseCode('Expected a 200 response code.');
        }
    }

    public function getAccount(string $accountId): EmailAccount
    {
        return new GuzzleEmailAccount($accountId, $this->httpClient);
    }
}
