<?php

declare(strict_types=1);

namespace DawidRza\Component\EmailClient\Symfony;

use DateTimeImmutable;
use DawidRza\Component\EmailClient\EmailAccount;
use DawidRza\Component\EmailClient\Exception\InvalidMessageException;
use DawidRza\Component\EmailClient\Exception\TransportException;
use DawidRza\Component\EmailClient\Service\EmailClientTransformer;
use DawidRza\Component\EmailClient\Type\Folder;
use DawidRza\Component\EmailClient\Type\Message;
use DawidRza\Component\EmailClient\Type\Received\ReceivedMessage;
use Iterator;

final class GuzzleEmailAccount implements EmailAccount
{
    /** @var GuzzleHttpClient */
    private $httpClient;

    /** @var string */
    private $accountId;

    public function __construct(
        string $accountId,
        GuzzleHttpClient $httpClient
    ) {
        $this->accountId = $accountId;
        $this->httpClient = $httpClient;
    }

    public function getFolder(string $folderName): Folder
    {
        $response = $this->httpClient->sendRequest(
            'GET',
            "/accounts/{$this->accountId}/folders/$folderName",
        );

        switch ($response['statusCode']) {
            case 200:
                return EmailClientTransformer::decomposeBodyFromGetFolder($response['responseBody']);
            default:
                throw TransportException::unexpectedResponseCode('Expected a 200 response code.');
        }
    }

    /**
     * @return Iterator<int, Folder>
     */
    public function getFolders(): iterable
    {
        $response = $this->httpClient->sendRequest(
            'GET',
            "/accounts/{$this->accountId}/folders",
        );

        switch ($response['statusCode']) {
            case 200:
                return EmailClientTransformer::decomposeBodyFromGetFolders($response['responseBody']);
            default:
                throw TransportException::unexpectedResponseCode('Expected a 200 response code.');
        }
    }

    /**
     * @return Iterator<int, ReceivedMessage>
     */
    public function getMessages(
        string $folderName,
        ?int $startingUID = null,
        ?int $limit = null
    ): iterable
    {
        $response = $this->httpClient->sendRequest(
            'GET',
            "/accounts/{$this->accountId}/folders/$folderName/messages",
            [
                'startingUID' => $startingUID,
                'limit' => $limit,
            ],
        );

        switch ($response['statusCode']) {
            case 200:
                return EmailClientTransformer::decomposeBodyFromGetMessages($response['responseBody']);
            default:
                throw TransportException::unexpectedResponseCode('Expected a 200 response code.');
        }
    }

    public function sendEmail(Message $message, ?DateTimeImmutable $dispatchedAt = null): void
    {
        if ($message->sender === null) {
            throw new InvalidMessageException('The message has no sender.');
        }

        if (count($message->recipients) === 0) {
            throw new InvalidMessageException('The message has no recipients.');
        }

        if (count($message->body) === 0) {
            throw new InvalidMessageException('The message has no body.');
        }

        $response = $this->httpClient->sendRequest(
            'POST',
            "/accounts/{$this->accountId}/messages",
            [],
            EmailClientTransformer::composeBodyForSendEmail($this->accountId, $message, $dispatchedAt),
        );

        if ($response['statusCode'] !== 202) {
            throw TransportException::unexpectedResponseCode('Expected a 202 response code.');
        }
    }
}
