<?php

declare(strict_types=1);

namespace DawidRza\Component\EmailClient\Service;

use DateTimeImmutable;
use DawidRza\Component\EmailClient\Exception\TransportException;
use DawidRza\Component\EmailClient\Type\Account\IncomingServer;
use DawidRza\Component\EmailClient\Type\Account\OutgoingServer;
use DawidRza\Component\EmailClient\Type\Address;
use DawidRza\Component\EmailClient\Type\Attachment;
use DawidRza\Component\EmailClient\Type\Content;
use DawidRza\Component\EmailClient\Type\Discovery\DiscoveryInformation;
use DawidRza\Component\EmailClient\Type\Discovery\ServerInformation;
use DawidRza\Component\EmailClient\Type\Folder;
use DawidRza\Component\EmailClient\Type\Message;
use DawidRza\Component\EmailClient\Type\Received\ReceivedAttachment;
use DawidRza\Component\EmailClient\Type\Received\ReceivedHeader;
use DawidRza\Component\EmailClient\Type\Received\ReceivedMessage;
use InvalidArgumentException;
use Iterator;

/**
 * @phpstan-type AddressRepresentation array{
 *     email: string,
 *     title: ?string,
 * }
 *
 * @phpstan-type AttachmentRepresentation array{
 *     filename: string,
 *     inline: bool,
 *     link: string,
 * }
 *
 * @phpstan-type ContentRepresentation array{
 *     content: string,
 *     type: string,
 * }
 *
 * @phpstan-type HeadersRepresentation array<string, string|list<string>>
 *
 * @phpstan-type MessageRepresentation array{
 *     attachments?: list<AttachmentRepresentation>,
 *     body: list<ContentRepresentation>,
 *     dispatchDate?: string,
 *     headers?: HeadersRepresentation,
 *     recipients: list<AddressRepresentation>,
 *     sender: AddressRepresentation,
 *     subject: ?string,
 * }
 *
 * @phpstan-type DiscoveryServer array{
 *     protocol: string,
 *     host: string,
 *     port: int,
 *     username: string,
 *     password: string,
 * }
 *
 * @phpstan-type RegisterAccountBody array{
 *     incomingServer: DiscoveryServer,
 *     outgoingServer: DiscoveryServer,
 * }
 */
final class EmailClientTransformer
{
    /**
     * @return RegisterAccountBody
     */
    public static function composeBodyForRegisterAccount(
        IncomingServer $incomingServer,
        OutgoingServer $outgoingServer
    ): array
    {
        return [
            'incomingServer' => [
                'protocol' => $incomingServer->protocol,
                'host' => $incomingServer->host,
                'port' => $incomingServer->port,
                'username' => $incomingServer->username,
                'password' => $incomingServer->password,
            ],
            'outgoingServer' => [
                'protocol' => $outgoingServer->protocol,
                'host' => $outgoingServer->host,
                'port' => $outgoingServer->port,
                'username' => $outgoingServer->username,
                'password' => $outgoingServer->password,
            ],
        ];
    }

    /**
     * @return MessageRepresentation
     */
    public static function composeBodyForSendEmail(
        string $accountId,
        Message $message,
        ?DateTimeImmutable $dispatchedAt
    ): array
    {
        $requestBody = [
            'accountId' => $accountId,
            'body' => array_map(
                static function (Content $content): array {
                    return [
                        'content' => $content->content,
                        'type' => $content->type,
                    ];
                },
                array_values($message->body),
            ),
            'recipients' => array_map(
                static function (Address $recipient): array {
                    return [
                        'email' => $recipient->email,
                        'title' => $recipient->title,
                    ];
                },
                array_values($message->recipients),
            ),
            'sender' => [
                'email' => $message->sender->email,
                'title' => $message->sender->title,
            ],
            'subject' => $message->subject,
        ];

        if (count($message->attachments) !== 0) {
            $requestBody['attachments'] = array_map(
                static function (Attachment $attachment): array {
                    return [
                        'filename' => $attachment->filename,
                        'link' => $attachment->url,
                        'inline' => $attachment->inline,
                    ];
                },
                array_values($message->attachments),
            );
        }

        if ($dispatchedAt !== null) {
            $requestBody['dispatchedAt'] = $dispatchedAt->format(DATE_ATOM);
        }

        if (count($message->headers) !== 0) {
            $requestBody['headers'] = $message->headers;
        }

        return $requestBody;
    }

    /**
     * @param array<mixed> $responseBody
     * @return DiscoveryInformation
     *
     * @throws TransportException
     */
    public static function decomposeBodyFromDiscoverConfiguration(array $responseBody): DiscoveryInformation
    {
        $crawler = ArrayCrawler::from($responseBody);

        $incomingServers = [];
        $outgoingServers = [];
        try {
            foreach ($crawler->crawlerList('incomingServers') as $serverCrawler) {
                $incomingServers[] = ServerInformation::create(
                    $serverCrawler->string('authenticationType'),
                    $serverCrawler->string('host'),
                    $serverCrawler->integer('port'),
                    $serverCrawler->string('protocolType'),
                    $serverCrawler->string('socketType'),
                    $serverCrawler->string('usernameType')
                );
            }

            foreach ($crawler->crawlerList('outgoingServers') as $serverCrawler) {
                $outgoingServers[] = ServerInformation::create(
                    $serverCrawler->string('authenticationType'),
                    $serverCrawler->string('host'),
                    $serverCrawler->integer('port'),
                    $serverCrawler->string('protocolType'),
                    $serverCrawler->string('socketType'),
                    $serverCrawler->string('usernameType')
                );
            }

            return DiscoveryInformation::create(
                $crawler->string('displayName'),
                $crawler->string('domain'),
                $incomingServers,
                $outgoingServers,
            );
        } catch (InvalidArgumentException $exception) {
            throw TransportException::unexpectedResponseFormat($exception->getMessage());
        }
    }

    /**
     * @param array<mixed> $responseBody
     *
     * @throws TransportException
     */
    public static function decomposeBodyFromGetFolder(array $responseBody): Folder
    {
        $crawler = ArrayCrawler::from($responseBody);

        $flags = [];
        try {
            $flagsCrawler = $crawler->crawler('flags');
            foreach ($flagsCrawler->keys() as $flagsKey) {
                $flags[] = $flagsCrawler->stringList($flagsKey);
            }

            return Folder::create(
                $crawler->string('name'),
                $flags,
                $crawler->integer('nextUID'),
                $crawler->integer('messageCount')
            );
        } catch (InvalidArgumentException $exception) {
            throw TransportException::unexpectedResponseFormat($exception->getMessage());
        }
    }

    /**
     * @param array<mixed> $responseBody
     * @return Iterator<int, Folder>
     *
     * @throws TransportException
     */
    public static function decomposeBodyFromGetFolders(array $responseBody): iterable
    {
        if (!array_key_exists('data', $responseBody) || !is_array($responseBody['data'])) {
            throw TransportException::unexpectedResponseFormat(sprintf(
                'Expected the key %s to exist',
                'data',
            ));
        }

        foreach ($responseBody['data'] as $folderKey => $folder) {
            if (!is_array($folder)) {
                throw TransportException::unexpectedResponseFormat(sprintf(
                    'Expected the key %s to be an array, got %s',
                    $folderKey,
                    gettype($folder),
                ));
            }

            yield self::decomposeBodyFromGetFolder($folder);
        }
    }

    /**
     * @param array<mixed> $responseBody
     * @return Iterator<int, ReceivedMessage>
     *
     * @throws TransportException
     */
    public static function decomposeBodyFromGetMessages(array $responseBody): iterable
    {
        foreach ($responseBody as $receivedMessageKey => $receivedMessage) {
            if (!is_array($receivedMessage)) {
                throw TransportException::unexpectedResponseFormat(sprintf(
                    'Expected the key %s to be an array, got %s',
                    $receivedMessageKey,
                    gettype($receivedMessage),
                ));
            }

            $crawler = ArrayCrawler::from($receivedMessage);

            $attachments = [];
            $body = [];
            $headers = [];
            $recipients = [];
            try {
                foreach ($crawler->crawlerList('attachments') as $attachment) {
                    $attachments[] = ReceivedAttachment::create(
                        $attachment->string('content'),
                        $attachment->string('filename'),
                    );
                }

                foreach ($crawler->crawlerList('body') as $content) {
                    $body[] = Content::create(
                        $content->string('content'),
                        $content->string('type'),
                    );
                }

                $headersCrawler = $crawler->crawler('headers');
                foreach ($headersCrawler->keys() as $headerKey) {
                    if (!is_string($headerKey)) {
                        throw new InvalidArgumentException(sprintf(
                            "Expected the key $headerKey to be a string, got %s",
                            gettype($headerKey),
                        ));
                    }

                    /* TODO: Add validation here. */
                    /** @var null|string|list<string> $value */
                    $value = $headersCrawler->get($headerKey);

                    $headers[] = ReceivedHeader::create(
                        $headerKey,
                        $value,
                    );
                }

                $date = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', $crawler->string('date'));
                if (!$date instanceof DateTimeImmutable) {
                    throw new InvalidArgumentException('Failed to convert key date to a date.');
                }

                foreach ($crawler->crawlerList('recipients') as $recipient) {
                    $recipients[] = Address::create(
                        $recipient->string('email'),
                        $recipient->nullableString('title'),
                    );
                }

                $senderCrawler = $crawler->crawler('sender');

                yield ReceivedMessage::create(
                    $crawler->integer('UID'),
                    $attachments,
                    $body,
                    $crawler->stringList('flags'),
                    $headers,
                    Address::create(
                        $senderCrawler->string('email'),
                        $senderCrawler->nullableString('title'),
                    ),
                    $date,
                    $recipients,
                    $crawler->nullableString('subject'),
                );

            } catch (InvalidArgumentException $exception) {
                throw TransportException::unexpectedResponseFormat($exception->getMessage());
            }
        }
    }

    /**
     * @param array<mixed> $responseBody
     *
     * @throws TransportException
     */
    public static function decomposeBodyFromRegisterAccount(array $responseBody): string
    {
        $crawler = ArrayCrawler::from($responseBody);

        try {
            return $crawler->string('accountId');
        } catch (InvalidArgumentException $exception) {
            throw TransportException::unexpectedResponseFormat($exception->getMessage());
        }
    }
}
