<?php

declare(strict_types=1);

namespace DawidRza\Component\EmailClient\Service;

use InvalidArgumentException;

final class ArrayCrawler
{
    /** @var array<mixed> */
    private $data;

    /**
     * @param array<mixed> $data
     */
    private function __construct(
        array $data
    ) {
        $this->data = $data;
    }

    /**
     * @param array<mixed> $data
     */
    public static function from(array $data): self
    {
        return new self($data);
    }

    /**
     * @param int|string $key
     * @return array<mixed>
     */
    public function array($key): array
    {
        $value = $this->get($key);
        if (!is_array($value)) {
            throw new InvalidArgumentException(sprintf(
                'Expected the key %s to be an array, got %s',
                $key,
                gettype($value),
            ));
        }

        return $value;
    }

    /**
     * @param int|string $key
     */
    public function contains($key): bool
    {
        return array_key_exists($key, $this->data);
    }

    /**
     * @param int|string $key
     */
    public function crawler($key): self
    {
        $value = $this->get($key);
        if (!is_array($value)) {
            throw new InvalidArgumentException(sprintf(
                'Expected the key %s to be an array, got %s',
                $key,
                gettype($value),
            ));
        }

        return self::from($value);
    }

    /**
     * @param int|string $key
     * @return list<self>
     */
    public function crawlerList($key): array
    {
        $crawler = $this->crawler($key);

        return array_map(
            static function ($key) use ($crawler): ArrayCrawler  { return $crawler->crawler($key); },
            array_keys($crawler->data),
        );
    }

    /**
     * @param int|string $key
     * @return mixed
     */
    public function get($key)
    {
        if (!$this->contains($key)) {
            throw new InvalidArgumentException(sprintf(
                'Expected the key %s to be exist.',
                $key,
            ));
        }

        return $this->data[$key];
    }

    /**
     * @param int|string $key
     */
    public function integer($key): int
    {
        $value = $this->get($key);
        if (!is_int($value)) {
            throw new InvalidArgumentException(sprintf(
                'Expected the key %s to be an integer, got %s',
                $key,
                gettype($value),
            ));
        }

        return $value;
    }

    /**
     * @return list<int|string>
     */
    public function keys(): array
    {
        return array_keys($this->data);
    }

    /**
     * @param int|string $key
     */
    public function nullableString($key): ?string
    {
        $value = $this->get($key);
        if (!is_string($value) && $value !== null) {
            throw new InvalidArgumentException(sprintf(
                'Expected the key %s to be a nullable string, got %s',
                $key,
                gettype($value),
            ));
        }

        return $value;
    }

    /**
     * @param int|string $key
     */
    public function string($key): string
    {
        $value = $this->get($key);
        if (!is_string($value)) {
            throw new InvalidArgumentException(sprintf(
                'Expected the key %s to be a string, got %s',
                $key,
                gettype($value),
            ));
        }

        return $value;
    }

    /**
     * @param int|string $key
     * @return list<string>
     */
    public function stringList($key): array
    {
        $crawler = $this->crawler($key);

        return array_map(
            static function ($key) use ($crawler): string { return $crawler->string($key); },
            array_keys($crawler->data),
        );
    }
}
